package com.xebialabs.deployit.plugin.generic.deployed;

import com.google.common.collect.Maps;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.*;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.step.InspectScriptExecutionStep;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Sets.newHashSet;

@SuppressWarnings("serial")
@Metadata(description = "Abstract deployed that can target any deployable to a generic container")
public abstract class AbstractDeployed<D extends Deployable> extends BaseDeployed<D, HostContainer> {

    private Map<String, Object> freeMarkerContext = Collections.singletonMap("deployed", (Object) this);

    //Injected by the DeployedInjector during pre-planning phase
    //----------------------------------------------------------
    private DeployedApplication deployedApplication;
    private Operation planOperation;
    //----------------------------------------------------------

    @Property(defaultValue = "false", hidden = true, description = "The generic container requires a restart for the action performed by this deployed.", required = false)
    private boolean restartRequired;

    @Property(defaultValue = "false", hidden = true, description = "The generic container requires a restart for the NOOP action performed by this deployed.", required = false)
    private boolean restartRequiredForNoop;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the create operation.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "40", description = "The order of the step in the step list for the destroy operation.")
    private int destroyOrder;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the modify operation.")
    private int modifyOrder;

    @Property(hidden = true, defaultValue = "50", description = "The order of the step in the step list for the noop operation.")
    private int noopOrder;

    @Property(hidden = true, defaultValue = "Create")
    private String createVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "Destroy")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "Modify")
    private String noopVerb;

    @Property(required = false, hidden = true, description = "Classpath to the script used to inspect the generic container.")
    private String inspectScript;

    @Property(hidden = true, required = false, description = "Additional classpath resources that should be uploaded to the working directory before executing the inspect script.")
    private Set<String> inspectClasspathResources = newHashSet();

    @Property(hidden = true, required = false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the inspect script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> inspectTemplateClasspathResources = newHashSet();


    @SuppressWarnings("rawtypes")
    public List<Step> controlTaskDispatch(String name) {
        String scriptPropertyName = name + "Script";
        PropertyDescriptor propertyDescriptor = DescriptorRegistry.getDescriptor(getType()).getPropertyDescriptor(scriptPropertyName);
        checkArgument(propertyDescriptor != null, "Control task script property %s not defined for CI type %s", scriptPropertyName, getType());
        String scriptName = resolveExpression((String) propertyDescriptor.get(this));

        Map<String, Object> vars = Maps.newHashMap();
        vars.put("container", getContainer());
        vars.put("deployed", this);
        return Collections.<Step>singletonList(new ScriptExecutionStep(1, scriptName, getContainer(),
                vars, getDescription(name)));
    }

    @Inspect
    public void inspectContainer(InspectionPlanningContext ctx) {
        if (emptyToNull(getInspectScript()) != null) {
            InspectScriptExecutionStep step = new InspectScriptExecutionStep(this, getInspectScript(), getContainer(), freeMarkerContext, "Inspect " + this);
            step.setTemplateClasspathResources(newArrayList(getInspectTemplateClasspathResources()));
            step.setClasspathResources(newArrayList(getInspectClasspathResources()));
            ctx.addStep(step);
        }
    }

    public boolean isRestartRequired() {
        return restartRequired;
    }

    public void setRestartRequired(boolean restartRequired) {
        this.restartRequired = restartRequired;
    }

    public int getCreateOrder() {
        return createOrder;
    }

    public void setCreateOrder(int createOrder) {
        this.createOrder = createOrder;
    }

    public int getDestroyOrder() {
        return destroyOrder;
    }

    public void setDestroyOrder(int destroyOrder) {
        this.destroyOrder = destroyOrder;
    }

    public int getModifyOrder() {
        return modifyOrder;
    }

    public void setModifyOrder(int modifyOrder) {
        this.modifyOrder = modifyOrder;
    }

    public String getCreateVerb() {
        return createVerb;
    }

    public void setCreateVerb(String createVerb) {
        this.createVerb = createVerb;
    }

    public String getModifyVerb() {
        return modifyVerb;
    }

    public void setModifyVerb(String modifyVerb) {
        this.modifyVerb = modifyVerb;
    }

    public String getDestroyVerb() {
        return destroyVerb;
    }

    public void setDestroyVerb(String destroyVerb) {
        this.destroyVerb = destroyVerb;
    }

    public int getNoopOrder() {
        return noopOrder;
    }

    public void setNoopOrder(int noopOrder) {
        this.noopOrder = noopOrder;
    }

    public String getNoopVerb() {
        return noopVerb;
    }

    public void setNoopVerb(String noopVerb) {
        this.noopVerb = noopVerb;
    }

    public boolean isRestartRequiredForNoop() {
        return restartRequiredForNoop;
    }

    public void setRestartRequiredForNoop(boolean restartRequiredForNoop) {
        this.restartRequiredForNoop = restartRequiredForNoop;
    }

    public String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getDeployable().getName(), getContainer().getName());
    }

    public Map<String, Object> getDeployedAsFreeMarkerContext() {
        return freeMarkerContext;
    }

    public Map<String, String> resolveExpression(Map<String, String> expressions) {
        return ConfigurationHolder.resolveExpression(expressions, getDeployedAsFreeMarkerContext());
    }

    public Set<String> resolveExpression(Set<String> expressions) {
        return ConfigurationHolder.resolveExpression(expressions, getDeployedAsFreeMarkerContext());
    }

    public String resolveExpression(String expression) {
        return ConfigurationHolder.resolveExpression(expression, getDeployedAsFreeMarkerContext());
    }

    public Object getPropertyOrContainerDefault(String property) {
        checkNotNull(property, "Property must not be null.");
        Object o = getProperty(property, this);
        if (o == null) {
            o = getProperty(property, this.getContainer());
        }

        //checkNotNull(property, "Property %s not found in %s or in %s.", property, getType(), getContainer().getType());
        return o;
    }

    private Object getProperty(String property, ConfigurationItem ci) {
        Descriptor descriptor = DescriptorRegistry.getDescriptor(ci.getType());
        PropertyDescriptor propertyDescriptor = descriptor.getPropertyDescriptor(property);
        if (propertyDescriptor == null) {
            return null;
        }

        return propertyDescriptor.get(ci);
    }

    public DeployedApplication getDeployedApplication() {
        return deployedApplication;
    }

    public void setDeployedApplication(DeployedApplication deployedApplication) {
        this.deployedApplication = deployedApplication;
    }

    public Operation getPlanOperation() {
        return planOperation;
    }

    public void setPlanOperation(Operation planOperation) {
        this.planOperation = planOperation;
    }

    public String getInspectScript() {
        return resolveExpression(inspectScript);
    }

    public void setInspectScript(String inspectScript) {
        this.inspectScript = inspectScript;
    }

    public Set<String> getInspectClasspathResources() {
        return resolveExpression(inspectClasspathResources);
    }

    public void setInspectClasspathResources(Set<String> inspectClasspathResources) {
        this.inspectClasspathResources = inspectClasspathResources;
    }

    public Set<String> getInspectTemplateClasspathResources() {
        return resolveExpression(inspectTemplateClasspathResources);
    }

    public void setInspectTemplateClasspathResources(Set<String> inspectTemplateClasspathResources) {
        this.inspectTemplateClasspathResources = inspectTemplateClasspathResources;
    }

}
