/*
 * @(#)ExecutedSqlScript.java     1 Sep 2011
 *
 * Copyright © 2010 Andrew Phillips.
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */
package com.xebialabs.deployit.plugin.generic.deployed;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;

import java.io.File;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Function;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Ordering;
import com.google.common.collect.Sets;
import com.google.common.io.PatternFilenameFilter;
import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifacts;
import com.xebialabs.deployit.plugin.api.udm.artifact.DerivedArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.PlaceholderReplacer;
import com.xebialabs.deployit.plugin.api.validation.Placeholders;
import com.xebialabs.deployit.plugin.generic.ci.Folder;
import com.xebialabs.deployit.plugin.generic.freemarker.ConfigurationHolder;
import com.xebialabs.deployit.plugin.generic.step.ScriptExecutionStep;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.local.LocalFile;

@SuppressWarnings("serial")
@Metadata(virtual = true, description = "Scripts in the folder are executed against a Container based on a naming convention")
@Placeholders
public class ExecutedFolder<D extends Folder> extends AbstractDeployed<D> implements DerivedArtifact<D> {

    private Map<String, Object> freeMarkerContext = Collections.singletonMap("deployed", (Object) this);

    @Property(required = false, category = "Placeholders", description = "A key/value pair mapping of placeholders in the deployed artifact to their values. Special values are <ignore> and <empty>")
    private Map<String, String> placeholders = newHashMap();

    private OverthereFile placeholderProcessedFile;

    @Property(description = "Name of the executor script that will be executed for each script found in the folder.")
    private String executorScript;

    @Property(description = "Regular expression used to identify a script in the folder.  A successful match should returns a single group to which the rollbackScriptPostfix can be appended" +
            " to inorder to find the associated rollback script or the script's dependent subfolder.  e.g.([0-9]*-.*)\\.sql")
    private String scriptRecognitionRegex;

    @Property(description = "Regular expression used to identify a rollback script in the folder. A successful match should returns a single group, ie the logical script name. e.g. [0-9]*-.*-rollback\\.sql")
    private String rollbackScriptRecognitionRegex;

    @Property(description = "A script's associated rollback script is derived by using the 1st group identified by the scriptRecognitionRegex and then appending this postfix to it." +
            " e.g give name '01-myscript.sql', regex '([0-9]*-.*)\\.sql' and rollback script postfix '-rollback.sql', we can derive the name of the associated rollback script  to be '01-myscript-rollback.sql'")
    private String rollbackScriptPostfix;

    @Property(hidden = true, defaultValue = "common", description = "Common folder that should be uploaded to the working directory.")
    private String commonScriptFolderName;

    @Property(hidden = true, required=false, description = "Additional classpath resources that should be uploaded to the working directory before executing the script.")
    private Set<String> classpathResources = newHashSet();

    @Property(hidden = true, required=false, description = "Additional template classpath resources that should be uploaded to the working directory before executing the script." +
            "The template is first rendered and the rendered content copied to a file, with the same name as the template, in the working directory.")
    private Set<String> templateClasspathResources = newHashSet();


    public String resolveExpression(String expression) {
        return ConfigurationHolder.resolveExpression(expression, freeMarkerContext);
    }

    public String getDescription(String script, String verb) {
        return String.format("%s %s on %s", verb, script, getContainer().getName());
    }

    @Create
    public void executeCreate(DeploymentPlanningContext ctx, Delta d) {
        addSteps(ctx, this, identifyAndOrderCreateScriptsInFolder(getDerivedArtifactAsFile()),getScriptRecognitionRegex());
    }

    protected void addSteps(DeploymentPlanningContext ctx, ExecutedFolder<D> deployed, List<File> scriptsToRun, String scriptNameRegex) {
        File folder = deployed.getDerivedArtifactAsFile();
        Pattern pattern = Pattern.compile(scriptNameRegex);
        File commonResources = new File(folder, deployed.getCommonScriptFolderName());
        for (File script : scriptsToRun) {
            ScriptExecutionStep step = newScriptExecutionStep(script.getName(), deployed);
            if (commonResources.exists()) {
                step.getFileResources().add(commonResources);
            }
            File dependentSubFolderResource = new File(folder, extractScriptPrefix(pattern, script.getName()));
            if (dependentSubFolderResource.exists()) {
                step.getFileResources().add(dependentSubFolderResource);
            }
            step.setTemplateClasspathResources(newArrayList(getTemplateClasspathResources()));
            step.setClasspathResources(newArrayList(getClasspathResources()));
            step.setArtifact(script);
            ctx.addStep(step);
        }
    }

    protected ScriptExecutionStep newScriptExecutionStep(String script, ExecutedFolder<D> deployed) {
        return new ScriptExecutionStep(deployed.getCreateOrder(), deployed.getExecutorScript(),
                deployed.getContainer(), deployed.freeMarkerContext, deployed.getDescription(script, deployed.getCreateVerb()));
    }

    protected String extractScriptPrefix(Pattern pattern, String scriptName) {
        Matcher matcher = pattern.matcher(scriptName);
        boolean matches = matcher.matches();
        checkArgument(matches && matcher.groupCount() == 1, "Script recognition regular expression '%s' run on script name '%s' must return a single group" +
                " to which the rollbackScriptPostfix can be appended to, to determine the associated rollback script or to find its dependent sub-directory", pattern.pattern(), scriptName);
        return matcher.group(1);
    }

    @SuppressWarnings("unchecked")
    @Modify
    public void executeModify(DeploymentPlanningContext ctx, Delta d) {
        ExecutedFolder<D> previousDeployed = (ExecutedFolder<D>) d.getPrevious();
        File previousFolder = previousDeployed.getDerivedArtifactAsFile();
        File currentFolder = getDerivedArtifactAsFile();
        // downgrades
        addSteps(ctx, previousDeployed, compareAndIdentifyOrderedRollbackScriptsToRunForDowngrade(previousFolder, currentFolder), getRollbackScriptRecognitionRegex());
        // upgrades
        addSteps(ctx, this, compareAndIdentifyOrderedScriptsToRunForUpgrade(previousFolder, currentFolder),getScriptRecognitionRegex());
    }

    @Destroy
    public void executeDestroy(DeploymentPlanningContext ctx, Delta d) {
        File currentFolder = getDerivedArtifactAsFile();
        addSteps(ctx, this, identifyAndOrderRollbackScriptsInFolder(currentFolder),getRollbackScriptRecognitionRegex());
    }


    protected File getDerivedArtifactAsFile() {
        checkNotNull(getFile(), "%s has a null file property", this);
        checkArgument(getFile() instanceof LocalFile, "%s has a file that is not a LocalFile but a %s", this, getFile().getClass().getName());

        LocalFile localFile = (LocalFile) getFile();
        return localFile.getFile();
    }

    protected List<File> identifyAndOrderCreateScriptsInFolder(File folder) {
        List<File> scriptsToRun = findScriptsToRun(folder, getScriptRecognitionRegex());
        return Ordering.from(new FilenameComparator()).sortedCopy(scriptsToRun);
    }

    protected List<File> findScriptsToRun(File folder, String pattern) {
        File[] scriptsToRun = folder.listFiles(new PatternFilenameFilter(pattern));
        scriptsToRun = scriptsToRun == null ? new File[0] : scriptsToRun;
        return newArrayList(scriptsToRun);
    }

    protected List<File> identifyAndOrderRollbackScriptsInFolder(File folder) {
        List<File> scriptsToRun = findScriptsToRun(folder,getRollbackScriptRecognitionRegex());
        return Ordering.from(new FilenameComparator()).reverse().sortedCopy(scriptsToRun);
    }

    protected List<File> compareAndIdentifyOrderedScriptsToRunForUpgrade(File previousFolder, File currentFolder) {
        List<File> previousScripts = identifyAndOrderCreateScriptsInFolder(previousFolder);
        List<File> currentScripts = identifyAndOrderCreateScriptsInFolder(currentFolder);
        return Ordering.from(new FilenameComparator()).sortedCopy(difference(previousScripts, currentScripts));
    }

    protected List<File> compareAndIdentifyOrderedRollbackScriptsToRunForDowngrade(File previousFolder, File currentFolder) {
        List<File> previousScripts = identifyAndOrderCreateScriptsInFolder(previousFolder);
        List<File> currentScripts = identifyAndOrderCreateScriptsInFolder(currentFolder);
        Collection<File> missingScriptsInCurrentFolder = difference(currentScripts, previousScripts);
        List<File> rollbackScripts = newArrayList();
        Pattern pattern = Pattern.compile(getScriptRecognitionRegex());
        for (File missingScript : missingScriptsInCurrentFolder) {
            String rollbackScriptName = extractScriptPrefix(pattern, missingScript.getName()) + getRollbackScriptPostfix();
            File rollbackScript = new File(previousFolder, rollbackScriptName);
            if (rollbackScript.exists()) {
                rollbackScripts.add(rollbackScript);
            }
        }
        return Ordering.from(new FilenameComparator()).reverse().sortedCopy(rollbackScripts);
    }

    protected Collection<File> difference(List<File> previousFiles, List<File> currentFiles) {
        Set<FileNameEqualityWrapper> wrappedPreviousFiles = transform(previousFiles);
        Set<FileNameEqualityWrapper> wrappedCurrentFiles = transform(currentFiles);
        return Collections2.transform(Sets.difference(wrappedCurrentFiles, wrappedPreviousFiles), new Function<FileNameEqualityWrapper, File>() {
            @Override
            public File apply(FileNameEqualityWrapper input) {
                return input.getFile();
            }
        });
    }

    protected Set<FileNameEqualityWrapper> transform(List<File> files) {
        return newHashSet(Lists.transform(files, new Function<File, FileNameEqualityWrapper>() {
            @Override
            public FileNameEqualityWrapper apply(File input) {
                return new FileNameEqualityWrapper(input);
            }
        }));
    }


    public String getScriptRecognitionRegex() {
        return scriptRecognitionRegex;
    }

    public void setScriptRecognitionRegex(String scriptRecognitionRegex) {
        this.scriptRecognitionRegex = scriptRecognitionRegex;
    }

    public String getRollbackScriptRecognitionRegex() {
        return rollbackScriptRecognitionRegex;
    }

    public void setRollbackScriptRecognitionRegex(String rollbackScriptRecognitionRegex) {
        this.rollbackScriptRecognitionRegex = rollbackScriptRecognitionRegex;
    }

    public String getCommonScriptFolderName() {
        return resolveExpression(commonScriptFolderName);
    }

    public void setCommonScriptFolderName(String commonScriptFolderName) {
        this.commonScriptFolderName = commonScriptFolderName;
    }

    public String getExecutorScript() {
        return resolveExpression(executorScript);
    }

    public void setExecutorScript(String executorScript) {
        this.executorScript = executorScript;
    }

    public String getRollbackScriptPostfix() {
        return resolveExpression(rollbackScriptPostfix);
    }

    public void setRollbackScriptPostfix(String rollbackScriptPostfix) {
        this.rollbackScriptPostfix = rollbackScriptPostfix;
    }

    public Set<String> getClasspathResources() {
        return resolveExpression(classpathResources);
    }

    public void setClasspathResources(Set<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    public Set<String> getTemplateClasspathResources() {
        return resolveExpression(templateClasspathResources);
    }

    public void setTemplateClasspathResources(Set<String> templateClasspathResources) {
        this.templateClasspathResources = templateClasspathResources;
    }

    @Override
    public D getSourceArtifact() {
        return getDeployable();
    }

    @Override
    public void initFile(PlaceholderReplacer replacer) {
        Artifacts.replacePlaceholders(this, replacer);
    }

    @Override
    public Map<String, String> getPlaceholders() {
        return placeholders;
    }

    @Override
    public void setPlaceholders(Map<String, String> placeholders) {
        this.placeholders = placeholders;
    }

    @Override
    public OverthereFile getFile() {
        return placeholderProcessedFile;
    }

    @Override
    public void setFile(OverthereFile file) {
        this.placeholderProcessedFile = file;
    }

    public static class FilenameComparator implements Comparator<File> {
        @Override
        public int compare(File o1, File o2) {
            return o1.getName().compareTo(o2.getName());
        }
    }

    public static class FileNameEqualityWrapper {
        private final File file;

        public FileNameEqualityWrapper(File file) {
            this.file = file;
        }

        public File getFile() {
            return file;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            FileNameEqualityWrapper that = (FileNameEqualityWrapper) o;

            if (file != null ? !file.getName().equals(that.file.getName()) : that.file != null) return false;

            return true;
        }

        @Override
        public int hashCode() {
            return file != null ? file.getName().hashCode() : 0;
        }
    }

     protected final Logger logger = LoggerFactory.getLogger(this.getClass());
}
