package com.xebialabs.deployit.plugin.generic.step;

import java.io.UnsupportedEncodingException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static com.google.common.collect.Sets.newHashSet;
import static java.net.URLDecoder.decode;

@SuppressWarnings("serial")
public class InspectScriptExecutionStep extends BaseExecutionStep {

    private static final String defaultEncoding = "UTF-8";

    protected ConfigurationItem inspectedItem;

    public static final String INSPECTED_PROPERTY_PRELUDE = "INSPECTED:";

    public static final String DISCOVERED_ITEM_PRELUDE = "DISCOVERED:";

    public InspectScriptExecutionStep(ConfigurationItem inspectedItem, String scriptPath, HostContainer container, Map<String, Object> vars, String description) {
        super(scriptPath, container, vars, description);
        this.inspectedItem = inspectedItem;
    }

    @Override
    public int getOrder() {
        return 50;
    }

    @Override
    public StepExitCode execute(final ExecutionContext ctx) throws Exception {
        StepExitCode result = handleExecute(new InspectionProtocolContext(ctx));
        return result;
    }

    protected boolean handleOutputLine(String line, ExecutionContext ctx) {
        int posOfEqualsSign = line.indexOf('=');
        if (posOfEqualsSign < 0) {
            return false;
        }

        try {
            if (line.startsWith(INSPECTED_PROPERTY_PRELUDE)) {
                String name = decode(line.substring(INSPECTED_PROPERTY_PRELUDE.length(), posOfEqualsSign), defaultEncoding);
                String value = line.substring(posOfEqualsSign + 1);
                return handleInspectedProperty(name, value, ctx);
            }

            if (line.startsWith(DISCOVERED_ITEM_PRELUDE)) {
                String id = decode(line.substring(DISCOVERED_ITEM_PRELUDE.length(), posOfEqualsSign), defaultEncoding);
                String typeName = line.substring(posOfEqualsSign + 1);
                return handleDiscoveredItem(id, typeName, ctx);
            }
        } catch (UnsupportedEncodingException encodingException) {
            logger.debug("Unsupported encoding exception was encountered while parsing inspected and / or discovered items");
            return false;
        }
        return false;
    }

    private boolean handleInspectedProperty(String name, String value, ExecutionContext ctx) throws UnsupportedEncodingException {
        PropertyDescriptor pd = inspectedItem.getType().getDescriptor().getPropertyDescriptor(name);
        if (pd == null) {
            ctx.logError("Inspected unknown property " + name);
            return false;
        }

        if (pd.getKind() == PropertyKind.SET_OF_CI) {
            Set<ConfigurationItem> refs = newHashSet(getCis(value, pd));
            pd.set(inspectedItem, refs);
        } else if (pd.getKind() == PropertyKind.SET_OF_STRING) {
            Set<String> stringProperties = newHashSet(getStrings(value));
            pd.set(inspectedItem, stringProperties);
        } else if (pd.getKind() == PropertyKind.LIST_OF_CI) {
            List<ConfigurationItem> refs = newArrayList(getCis(value, pd));
            pd.set(inspectedItem, refs);
        } else if (pd.getKind() == PropertyKind.LIST_OF_STRING) {
            List<String> stringProperties = newArrayList(getStrings(value));
            pd.set(inspectedItem, stringProperties);
        } else if (pd.getKind() == PropertyKind.MAP_STRING_STRING) {
            Map<String, String> mapProperties = newHashMap();
            for (String property : value.split(",")) {
                if (property.contains(":")) {
                    String[] keyValue = property.split(":");
                    mapProperties.put(decode(keyValue[0], defaultEncoding), decode(keyValue[1], defaultEncoding));
                }
            }
            pd.set(inspectedItem, mapProperties);
        } else {
            pd.set(inspectedItem, decode(value, defaultEncoding));
        }

        logger.debug("Inspected property {} with value {}", name, value);
        return true;
    }

    private static Collection<String> getStrings(String value) throws UnsupportedEncodingException {
        Collection<String> stringProperties = newArrayList();
        if(value.length() > 0) {
            for (String property : value.split(",")) {
                stringProperties.add(decode(property, defaultEncoding));
            }
        }
        return stringProperties;
    }

    private static Collection<ConfigurationItem> getCis(String value, PropertyDescriptor pd) throws UnsupportedEncodingException {
        Collection<ConfigurationItem> refs = newArrayList();
        if(value.length() > 0) {
            Descriptor descriptor = DescriptorRegistry.getDescriptor(pd.getReferencedType());
            for (String id : value.split(",")) {
                ConfigurationItem ref = descriptor.newInstance();
                ref.setId(decode(id, defaultEncoding));
                refs.add(ref);
            }
        }
        return refs;
    }

    private boolean handleDiscoveredItem(String id, String typeName, ExecutionContext ctx) throws UnsupportedEncodingException {
        Type type = Type.valueOf(decode(typeName, defaultEncoding));
        if (!DescriptorRegistry.exists(type)) {
            ctx.logError("Discovered item " + id + " of unknown type " + type);
            return false;
        }
        Descriptor d = DescriptorRegistry.getDescriptor(type);
        ConfigurationItem discoveredItem = d.newInstance();
        discoveredItem.setId(id);

        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            if (isReferenceToParent(pd)) {
                pd.set(discoveredItem, inspectedItem);
            }
        }

        ctx.getInspectionContext().discovered(discoveredItem);
        logger.debug("Discovered item {} with type {}", id, type);
        return true;
    }

    protected boolean isReferenceToParent(PropertyDescriptor pd) {
        return pd.getKind() == PropertyKind.CI && pd.isAsContainment() && inspectedItem.getType().getDescriptor().isAssignableTo(pd.getReferencedType());
    }

    private class InspectionProtocolContext implements ExecutionContext {

        private final ExecutionContext context;

        public InspectionProtocolContext(ExecutionContext context) {
            this.context = context;
        }

        @Override
        public void logOutput(String output) {
            if (handleOutputLine(output, context)) {
                logger.trace("Handled inspection protocol line {}", output);
            } else {
                context.logOutput(output);
            }
        }

        @Override
        public void logError(String error) {
            context.logError(error);
        }

        @Override
        public void logError(String error, Throwable t) {
            context.logError(error, t);
        }

        @Override
        public Object getAttribute(String name) {
            return context.getAttribute(name);
        }

        @Override
        public void setAttribute(String name, Object value) {
            context.setAttribute(name, value);
        }

        @Override
        public InspectionContext getInspectionContext() {
            return context.getInspectionContext();
        }

    }
}
