package com.xebialabs.deployit.plugin.generic.step;

import com.google.common.base.Preconditions;

import com.xebialabs.deployit.plugin.api.Deprecations;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;

@SuppressWarnings("serial")
public class ArtifactCopyStep extends BaseDeploymentStep implements StageableStep {

    private OverthereFile sourceFile;

    //This parameter makes sense only for direct java usage. No need to annotate with @StepParameter
    private String sourceFileDescription;

    private Artifact sourceArtifact;

    private String targetPath;

    private boolean createTargetPath;

    private String targetFileName;

    //Weird logic is behind this field. Do not expose as @StepParameter
    private boolean preserveExistingFiles;

    private StagedFile stagedSourceArtifact;

    public ArtifactCopyStep(int order, OverthereFile sourceFile, HostContainer container, String targetPath) {
        super(order, container);
        this.sourceFile = sourceFile;
        this.targetPath = targetPath;
        Preconditions.checkNotNull(targetPath);
    }

    @Override
    public void requestStaging(final StagingContext ctx) {
        if (sourceArtifact != null) {
            logger.debug("[{}] is requesting staging of [{}]", this, sourceArtifact);
            stagedSourceArtifact = ctx.stageArtifact(sourceArtifact, container.getHost());
        } else {
            logger.debug("[{}] is not requesting artifact staging.", this);
        }
    }

    @Override
    protected StepExitCode doExecute() throws Exception {
        if(stagedSourceArtifact != null){
            return doCopy(stagedSourceArtifact.get(getRemoteConnection(), getRemoteWorkingDirectory(), getCtx()));
        }else{
            return doCopy(resolveSourceFile());
        }
    }

    private StepExitCode doCopy(OverthereFile sourceFile) throws Exception{
        Preconditions.checkNotNull(targetFileName);

        OverthereFile remoteDir = getRemoteConnection().getFile(targetPath);

        if (!remoteDir.exists()) {
            if (createTargetPath) {
                getCtx().logOutput("Creating path " + targetPath + " on host " + getContainer().getHost());
                remoteDir.mkdirs();
            } else if (sourceFile.isFile()) {
                getCtx().logError("Path " + targetPath + " on host " + getContainer().getHost() + " does not exist.");
                return StepExitCode.FAIL;
            }
        }

        OverthereFile remoteFile;
        if (sourceFile.isDirectory()) {
            remoteFile = remoteDir;
        } else {
            remoteFile = remoteDir.getFile(getTargetFileName());
            if (remoteFile.exists()) {
                getCtx().logOutput(remoteFile.getPath() + " already exists on host " + getContainer().getHost() + ". Will replace.");
                remoteFile.delete();
            }
        }

        getCtx().logOutput("Copying " + sourceFile.getPath() + " to " + remoteFile.getPath() + " on host " + getContainer().getHost());
        sourceFile.copyTo(remoteFile);
        return StepExitCode.SUCCESS;
    }

    protected OverthereFile resolveSourceFile() {
        checkNotNull(sourceFile);
        return sourceFile;
    }

    @Override
    public String getDescription() {
        String description = super.getDescription();
        if (description == null) {
            return "Copy " + getSourceFileDescription() + " to " + getContainer().getHost();
        }
        return description;
    }

    public String getSourceFileDescription() {
        return sourceFileDescription;
    }

    public void setSourceFileDescription(final String sourceFileDescription) {
        this.sourceFileDescription = sourceFileDescription;
    }

    public Artifact getSourceArtifact() {
        return sourceArtifact;
    }

    public void setSourceArtifact(final Artifact sourceArtifact) {
        this.sourceArtifact = sourceArtifact;
    }

    public boolean isCreateTargetPath() {
        return createTargetPath;
    }

    public void setCreateTargetPath(final boolean createTargetPath) {
        this.createTargetPath = createTargetPath;
    }

    public String getTargetFileName() {
        return targetFileName;
    }

    public void setTargetFileName(final String targetFileName) {
        this.targetFileName = targetFileName;
    }

    public boolean isPreserveExistingFiles() {
        return preserveExistingFiles;
    }

    /**
     * Property is not used anymore in the CopyStep and will be ignored.
     */
    @Deprecated
    public void setPreserveExistingFiles(final boolean preserveExistingFiles) {
        Deprecations.deprecated("**Deprecated** Setting property 'preserveExistingFiles' on ArtifactCopyStep is deprecated and will be ignored.");
        this.preserveExistingFiles = preserveExistingFiles;
    }

    public void setDescription(String description) {
        super.setDescription(description);
    }

}
