package com.xebialabs.deployit.plugin.generic.planning;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

import java.util.List;

import static com.google.common.base.Predicates.and;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.collect.Lists.newLinkedList;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.*;

public abstract class SingleTypeContributor<D extends Deployed<?, ?>> {
    protected final Predicate<Delta> isOfType;

    protected List<D> deployedsCreated;
    protected List<TypedDelta> deployedsModified;
    protected List<D> deployedsRemoved;

    protected SingleTypeContributor(Class<? extends D> classOfDeployed) {
        isOfType = new IsSubtypeOf(Type.valueOf(classOfDeployed));
    }

    protected void filterDeltas(Deltas deltas, DeploymentPlanningContext ctx) {
        deployedsCreated = newLinkedList(transform(
                filter(deltas.getDeltas(), and(isOfType, operationIs(CREATE))),
                new Function<Delta, D>() {
                    @SuppressWarnings("unchecked")
                    @Override
                    public D apply(Delta input) {
                        return (D) input.getDeployed();
                    }
                }));
        deployedsModified = newLinkedList(transform(
                filter(deltas.getDeltas(), and(isOfType, operationIs(MODIFY))),
                new Function<Delta, TypedDelta>() {
                    @Override
                    public TypedDelta apply(Delta input) {
                        return new TypedDelta(input);
                    }
                }));
        deployedsRemoved = newLinkedList(transform(
                filter(deltas.getDeltas(), and(isOfType, operationIs(DESTROY))),
                new Function<Delta, D>() {
                    @SuppressWarnings("unchecked")
                    @Override
                    public D apply(Delta input) {
                        return (D) input.getPrevious();
                    }
                }));
    }

    protected static OperationEquals operationIs(Operation operationToMatch) {
        return new OperationEquals(operationToMatch);
    }

    protected static class OperationEquals implements Predicate<Delta> {
        private final Operation operationToMatch;

        protected OperationEquals(Operation operationToMatch) {
            this.operationToMatch = operationToMatch;
        }

        @Override
        public boolean apply(Delta input) {
            return input.getOperation().equals(operationToMatch);
        }
    }

    protected static class IsSubtypeOf implements Predicate<Delta> {
        private Type typeToMatch;

        protected IsSubtypeOf(Type typeToMatch) {
            this.typeToMatch = typeToMatch;
        }

        @Override
        public boolean apply(Delta input) {
            return getType(input).instanceOf(typeToMatch);
        }

        // move to DeltaUtils or whatever
        private static Type getType(Delta delta) {
            return (delta.getOperation().equals(DESTROY)
                    ? delta.getPrevious().getType()
                    : delta.getDeployed().getType());
        }
    }

    @SuppressWarnings("serial")
    protected class TypedDelta implements Delta {
        private final Delta delegate;

        private TypedDelta(Delta delegate) {
            this.delegate = delegate;
        }

        @Override
        public Operation getOperation() {
            return delegate.getOperation();
        }

        @SuppressWarnings("unchecked")
        @Override
        public D getPrevious() {
            return (D) delegate.getPrevious();
        }

        @SuppressWarnings("unchecked")
        @Override
        public D getDeployed() {
            return (D) delegate.getDeployed();
        }

        @Override
        public List<String> getIntermediateCheckpoints() {
            return delegate.getIntermediateCheckpoints();
        }
    }
}
