# Get provided website name or use the deployed name.
$websiteName = if($deployed.websiteName) { $deployed.websiteName } else { $deployed.name }

# Verify website name is not being changed.
if($previousDeployed) {
    $previousWebsiteName = if($previousDeployed.websiteName) { $previousDeployed.websiteName } else { $previousDeployed.name }
    if($websiteName -ne $previousWebsiteName) {
        Write-Host "Renaming a website is not supported. Undeploy and deploy the website instead."
        Exit 1
    }
}

# Verify the website has at least one binding specified, otherwise it won't be able to start later
if(!$deployed.bindings){
    Write-Host "Website must have at least one binding specified."
    Exit 1
}

# Create physical path if it doesn't exist yet.
if (!(Test-Path $deployed.physicalPath)) {
    Write-Host "Creating empty website physical path [$($deployed.physicalPath)]."
    if(!(New-Item $deployed.physicalPath -Type Directory -Force)) {
        Write-Host "Cannot create directory [$($deployed.physicalPath)]."
        Exit 1
    }
}

# Create bindings array.
$bindings = @()
foreach ($db in $deployed.bindings) {
    $binding = ([WMICLASS]"ROOT\MicrosoftIISv2:ServerBinding").CreateInstance()
    if ($db.ipAddress -ne "*") {
        $binding.IP = $db.ipAddress
    }
    $binding.Port = $db.port
    if ($db.hostHeader) {
        $binding.Hostname = $db.hostHeader
    }
    # See http://stackoverflow.com/questions/2240725/add-new-serverbinding-to-iis-6-site-using-wmi-and-powershell
    $bindings += $binding.psobject.baseobject
}

# Check whether website already exists and then either retrieve or create it.
$website = getWebsite $websiteName
if ($website) {
    Write-Host "Modifying existing website [$websiteName]."

    # Set physical path.
    $webdir = Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class IIsWebVirtualDirSetting -Filter "Name = '$($website.Name)/ROOT'"
    $webdir.Path = $deployed.physicalPath
    $webdir.Put() | Out-Null

    # Set bindings. 
    $website.ServerBindings = $bindings
} else {
    Write-Host "Creating new website [$websiteName]."

    $iis = Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class "IIsWebService"
    $iis.CreateNewSite($websiteName, $bindings, $deployed.physicalPath) | Out-Null

    $website = getWebsite $websiteName
    
    # Set default "Directory Security" properties: anonymous and NTLM.
    $website.AuthFlags = 5

    # Set default "Home directory" properties: read access and application name.
    $webdir = Get-WmiObject -Namespace "ROOT\MicrosoftIISv2" -Class IIsWebVirtualDirSetting -Filter "Name = '$($website.Name)/ROOT'"
    $webdir.AccessFlags = 513
    $webdir.AppFriendlyName = 'Default Application'
    $webdir.Put() | Out-Null
}

# Set application pool.
$website.AppPoolId = $deployed.applicationPoolName

# Save website.
$website.Put() | Out-Null

# Set .NET version.
if($deployed.dotNetVersion) {
    updateDotNetVersionForVirtualDirectory "$($website.Name)/Root" $deployed.container.dotNetFrameworkInstallationPath $deployed.dotNetVersion
}
