import sys
import re
import traceback
from java.util import Map
from java.util import List
from com.xebialabs.deployit.plugin.jbossdm.exception import CliScriptExit

debugCommandLoggingEnabled = False
ansi_escape = re.compile(r'(?:\x1B[@-_]|[\x80-\x9F])[0-?]*[ -/]*[@-~]')

class CommandBuilder:
    def __init__(self, deployed, basicCmdLine):
        self.deployed = deployed
        self.cmd = basicCmdLine

    def add(self, key, attr):
        val = self.deployed.getProperty(key)
        if isinstance(val, bool):
            if val:
                val = "true"
            else:
                val = "false"
        if val is not None:
            self.addStringArgument(key, val, attr)
        return self

    def addIfNotBlank(self, key, attr):
        val = self.deployed.getProperty(key)
        if val and len(val.strip()) > 0:
            self.addStringArgument(key, val, attr)
        return self

    def addIfNotDefaultIntValue(self, key, attr):
        val = self.deployed.getProperty(key)
        if val > 0:
            self.addIntegerArgument(val, attr)
        return self

    def addStringArgument(self, key, val, attr):
        if(self.isPassword(key)):
            self.cmd.addRaw(' %s="' % (attr))
            self.cmd.addPassword('%s' % (val))
            self.cmd.addRaw('"')
        else:
            self.cmd.addRaw(' %s="%s"' % (attr, val))

    def addIntegerArgument(self, val, attr):
        self.cmd.addRaw(" %s=%s" % (attr, val))

    def isPassword(self, key):
        return self.deployed.getType().getDescriptor().getPropertyDescriptor(key).isPassword()

def joinList(delimiter, list):
    return delimiter.join(str(x) for x in list)

def logErrorAndExit(msg):
    logError(msg)
    logError(joinList("\n",traceback.extract_stack()))
    raise CliScriptExit(msg)

def logOutputAndExit(msg):
    logOutput(msg)
    raise CliScriptExit(msg)

def logError(msg):
    step.ctx.logError(str(msg))

def logOutput(msg):
    step.ctx.logOutput(str(msg))

def checkResultForErrors(r):
    if isinstance(r, Map):
        if "outcome" in r and r["outcome"] == "failed":
            logErrorAndExit(str(r["failure-description"]))
        elif len(r.keySet()) == 1:
            firstKey = r.keySet().iterator().next()
            if isErrorKey(firstKey):
                logErrorAndExit(str(r[firstKey]))
        elif not r.containsKey('outcome'):
            logErrorAndExit("Can not handle processing of object %s " % str(r))
    elif isinstance(r, List):
        for i in r:
            checkResultForErrors(i)
    elif r:
        if isErrorResult(r):
            logErrorAndExit(r)

def isErrorKey(firstKey):
    return firstKey.startswith('JBAS') or firstKey.startswith('WFLY')

def isErrorResult(r):
    return re.search("JBAS[0-9]+|WFLY[A-Z0-9]+", r)

def executeCmd(cmd, exitOnFailure=True):
    if debugCommandLoggingEnabled:
        logOutput("Executing command:\n " + cmd)
    try:
        r = step.executeCliCommand(cmd)
    except:
        logErrorAndExit("Failed to execute command. Error %s with message %s " % (sys.exc_info()[0], sys.exc_info()[1]))

    if debugCommandLoggingEnabled:
        logOutput("Result:\n " + str(r))

    if exitOnFailure and r:
        checkResultForErrors(r)

    if r and isinstance(r, List):
        r = [i for i in r if i]
        if len(r) == 1:
            r = r[0]

    return r

def uploadFile(file, targetName):
    return step.uploadToWorkingDirectory(file, targetName)

def targetServerGroups(undeployment=False):
    return container.getTargetServerGroup(undeployment)

def multiSpaceDelimitedLineToList(line):
    if not line:
        return []
    line = line.strip()
    values = []
    for token in line.split(" "):
        if len(token.strip()) > 0:
            values.append(token.strip())
    return values

def discovered(container, name, type):
    return step.discovered("%s/%s" % (container.id,name), type)

def prependProfilePath(cmd):
    if str(container.mode) == "DOMAIN":
        return "/profile=%s%s" %(container.profile, cmd)
    return cmd

def toString(propVal):
    if isinstance(propVal, bool):
        if propVal:
            propVal = "true"
        else:
            propVal = "false"
    else:
        propVal = str(propVal)
    return propVal


def getReleaseVersion():
    return executeCmd("/:read-attribute(name=release-version)", False)["result"]

def getProductVersion():
    return getReplacedDict(executeCmd("/:read-attribute(name=product-version)", False).replace("=>",":"))["result"]

def getProductName():
    return getReplacedDict(executeCmd("/:read-attribute(name=product-name)", False).replace("=>",":"))["result"]

def getReplacedDict(propVal):
    return eval(ansi_escape.sub('', propVal))

def isEAP7():
    version = getProductVersion()
    name = getProductName()
    if version and version[0:2] in ["7."] and name and "EAP" in name:
        logOutput("EAP 7.0 detected.")
        return True
    return False



