package com.xebialabs.deployit.plugin.netscaler;

import java.util.Collection;
import java.util.Map;
import java.util.Set;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.DefaultOrders;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.validation.Range;
import com.xebialabs.deployit.plugin.generic.step.WaitStep;
import com.xebialabs.deployit.plugin.lb.ci.LoadBalancer;
import com.xebialabs.deployit.plugin.netscaler.step.CliScriptExecutionStep;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static java.lang.String.format;

@Metadata(root = Metadata.ConfigurationItemRoot.NESTED, virtual = false, description = "A Citrix NetScaler LoadBalancer")
public class NetScaler extends LoadBalancer {
    public static final String CONTAINER_NS_SHUTDOWN_DELAY = "netscalerShutdownDelay";
    private static final String CONTAINER_NS_ADDRESS = "netscalerAddress";

    @Property(defaultValue = "" + (DefaultOrders.STOP_ARTIFACTS - 2), hidden = true)
    private int disableServersOrder = DefaultOrders.STOP_ARTIFACTS - 2;

    @Property(defaultValue = "netscaler/disable-server.cli.ftl", hidden = true)
    private String disableServersScript = "netscaler/disable-server.cli.ftl";

    @Property(defaultValue = "" + (DefaultOrders.START_ARTIFACTS + 4), hidden = true)
    private int enableServersOrder = DefaultOrders.START_ARTIFACTS + 4;

    @Property(defaultValue = "netscaler/enable-server.cli.ftl", hidden = true)
    private String enableServersScript = "netscaler/enable-server.cli.ftl";

    @Property(required = false, description = "The amount of seconds to wait before the servers are disabled in the load balancer", defaultValue = "0")
    @Range(minimum = 0)
    private int defaultShutdownDelay;

    @Property(asContainment = true)
    private Host host;

    @Override
    public void stopTraffic(DeploymentPlanningContext ctx, Set<Container> serversToDisable) {
        validateAllServersHaveNetScalerAddress(serversToDisable);
        ctx.addStep(new CliScriptExecutionStep(disableServersOrder, "source", disableServersScript, this, createContext(serversToDisable), format("Disable servers %s in %s", serversToDisable, this)));
        ctx.addStep(new WaitStep(disableServersOrder, determineMaxWait(serversToDisable), this.toString(), "disable servers"));
    }

    @Override
    public void startTraffic(DeploymentPlanningContext ctx, Set<Container> serversToEnable) {
        validateAllServersHaveNetScalerAddress(serversToEnable);
        ctx.addStep(new CliScriptExecutionStep(enableServersOrder, "source", enableServersScript, this, createContext(serversToEnable), format("Enable servers %s in %s", serversToEnable, this)));
    }

    private Map<String, Object> createContext(Set<Container> servers) {
        return ImmutableMap.of("loadBalancer", this, "servers", servers);
    }

    @VisibleForTesting int determineMaxWait(Set<Container> servers) {
        int waitTime = 0;
        for (Container server : servers) {
            int serverTime = waitForServer(server);
            waitTime = serverTime > waitTime ? serverTime : waitTime;
        }
        return waitTime;
    }

    public int waitForServer(Container server) {
        int serverTime = server.<Integer>getProperty(CONTAINER_NS_SHUTDOWN_DELAY);
        if (serverTime < 0) {
            return defaultShutdownDelay;
        }
        return serverTime;
    }

    private void validateAllServersHaveNetScalerAddress(Collection<Container> servers) {
        for (Container server : servers) {
            serverAddress(server);
        }
    }

    public String serverAddress(Container server) {
        String serverAddress = server.<String>getProperty(CONTAINER_NS_ADDRESS);
        if (!Strings.nullToEmpty(serverAddress).trim().isEmpty()) {
            return serverAddress;
        }

        if (server.getType().instanceOf(Type.valueOf(HostContainer.class))) {
            return ((HostContainer) server).getHost().getProperty("address");
        }

        throw new IllegalStateException(format("Could not dermine netscaler Address for container [%s]", server));
    }

    @Override
    public Host getHost() {
        return host;
    }

    public void setDefaultShutdownDelay(int defaultShutdownDelay) {
        this.defaultShutdownDelay = defaultShutdownDelay;
    }
}
