package com.xebialabs.deployit.plugin.netscaler;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.overthere.HostContainer;

import static java.lang.String.format;

public class ServerOrService extends NetScalerItem {
    private final Container server;
    private int defaultShutdownDelay;

    ServerOrService(Container container, int defaultShutdownDelay) {
        this.server = container;
        this.defaultShutdownDelay = defaultShutdownDelay;
    }

    @Override
    public int getDelay() {
        int serverTime = server.<Integer>getProperty(NetScaler.CONTAINER_NS_SHUTDOWN_DELAY);
        if (serverTime < 0) {
            return defaultShutdownDelay;
        }
        return serverTime;
    }

    @Override
    public String getDelayString() {
        return "" + getDelay();
    }

    @Override
    public String getAddress() {
        String serverAddress = server.<String>getProperty(NetScaler.CONTAINER_NS_ADDRESS);
        if (!Strings.nullToEmpty(serverAddress).trim().isEmpty()) {
            return serverAddress;
        }

        if (server.getType().instanceOf(Type.valueOf(HostContainer.class))) {
            Host host = ((HostContainer) server).getHost();
            if (host.hasProperty("address")) {
                return host.getProperty("address");
            } else {
                logger.info("Host {} has no address property, falling back to 'localhost'", host);
                // Fallback scenario, overthere.LocalHost has no address property.
                return "localhost";
            }
        }

        throw new IllegalStateException(format("Could not dermine netscaler Address for server [%s]", server));
    }

    public String getNetScalerType() {
        return server.getProperty("netscalerType");
    }

    @Override
    public String toString() {
        return format("%s(%s)", getNetScalerType(), getAddress());
    }

    private static final Logger logger = LoggerFactory.getLogger(ServerOrService.class);
}
