/*
 * Copyright (c) 2008-2010 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.overthere;

import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.overthere.ConnectionOptions.OPERATING_SYSTEM;
import static com.xebialabs.overthere.ConnectionOptions.TEMPORARY_DIRECTORY_PATH;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.udm.Container;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.overthere.ConnectionOptions;
import com.xebialabs.overthere.OperatingSystemFamily;
import com.xebialabs.overthere.Overthere;
import com.xebialabs.overthere.OverthereConnection;

/**
 * A machine that runs middleware.
 */
@SuppressWarnings("serial")
@Metadata(root = Metadata.ConfigurationItemRoot.INFRASTRUCTURE, virtual = true, description = "A machine that runs middleware, on which scripts can be executed, etc.")
public class Host extends BaseConfigurationItem implements Container {

	@Property(description = "Protocol to use when connecting to this host", hidden = true)
	private String protocol;

	@Property(description = "Operating system")
	private OperatingSystemFamily os;

	@Property(description = "Directory into which temporary files are stored. Will be cleaned up when the connection is closed.", required = false)
	private String temporaryDirectoryPath;

	public OverthereConnection getConnection() {
		logger.debug("Using connection protocol {}", protocol);
		return Overthere.getConnection(protocol, getConnectionOptions());
	}

	private ConnectionOptions getConnectionOptions() {
		ConnectionOptions options = new ConnectionOptions();

		// set the operating system
		setConnectionOption(options, OPERATING_SYSTEM, getOs());

		// set the temporary directory path if it has been set
		if (!nullToEmpty(getTemporaryDirectoryPath()).trim().isEmpty()) {
			setConnectionOption(options, TEMPORARY_DIRECTORY_PATH, getTemporaryDirectoryPath());
		}

		// copy all synthetic properties to connection properties
		for (Map.Entry<String, Object> prop : getSyntheticProperties().entrySet()) {
			setConnectionOption(options, prop.getKey(), prop.getValue());
		}

		return options;
	}
	
	private void setConnectionOption(ConnectionOptions options, String key, Object value) {
		if(logger.isDebugEnabled())
			logger.debug("Setting connection option {}: {}", key, value);
		options.set(key, value);
	}

	public OperatingSystemFamily getOs() {
    	return os;
    }

	public void setOs(OperatingSystemFamily os) {
    	this.os = os;
    }

	public String getTemporaryDirectoryPath() {
		return temporaryDirectoryPath;
	}

	public void setTemporaryDirectoryPath(String temporaryDirectoryPath) {
		this.temporaryDirectoryPath = temporaryDirectoryPath;
	}

	private static Logger logger = LoggerFactory.getLogger(Host.class);

}
