package com.xebialabs.deployit.plugin.powershell;

import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Resources;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.overthere.HostContainer;
import com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.PowerShellScriptCallback;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.executePowerShellScript;
import static com.xebialabs.deployit.plugin.powershell.PowerShellStepUtils.previewPowerShellScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScripts;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.overthere.OperatingSystemFamily.WINDOWS;

@SuppressWarnings("serial")
abstract class PowerShellStep implements PreviewStep {

    private static final String BASE_RUNTIME_PATH = "powershell/runtime";

    private HostContainer container;
    Deployed<?, ?> deployed;
    private String scriptPath;
    private Map<String, Object> powerShellVars;
    private String description;
    private boolean uploadArtifactData = true;
    private List<String> classpathResources = newArrayList();
    private Preview preview;

    public PowerShellStep(HostContainer container, Deployed<?, ?> deployed, String scriptPath, Map<String, Object> powerShellVars, String description) {
        this.container = checkNotNull(container, "container is null");
        checkArgument(this.container.getHost().getOs() == WINDOWS, "PowerShell scripts can only be run on Windows hosts");
        this.deployed = deployed;
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.powerShellVars = checkNotNull(powerShellVars, "powerShellVars is null");
        this.description = checkNotNull(description, "description is null");
    }

    @Override
    public Preview getPreview() {
        if(preview == null) {
            return previewPowerShellScript(getPowerShellScriptCallback());
        }
        return preview;
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        return executePowerShellScript(container, ctx, getPowerShellScriptCallback());
    }

    private PowerShellScriptCallback getPowerShellScriptCallback() {
        return new PowerShellStepUtils.PowerShellScriptCallback() {
            @Override
            public String getScriptPath() {
                return scriptPath;
            }
            
            @Override
            public String generateScript(OverthereConnection conn) {
                return doGenerateScript(conn);
            }
            @Override
            public void uploadAdditionalResources(HostContainer container, ExecutionContext ctx, OverthereConnection conn) {
                doUploadClasspathResources(ctx, conn);
            }
            
        };
    }
    String doGenerateScript(OverthereConnection connection) {
        logger.debug("Starting generation of PowerShell script for step [{}]", getDescription());
        StringBuilder scriptBuilder = new StringBuilder();
        logger.debug("Appending runtime scripts");
        appendBaseRuntimeScript(scriptBuilder);
        logger.debug("Appending container runtime scripts");
        appendContainerRuntimeScripts(scriptBuilder);
        logger.debug("Appending container library scripts");
        appendContainerLibraryScripts(scriptBuilder);
        logger.debug("Appending deployed library scripts");
        appendDeployedLibraryScripts(scriptBuilder);
        logger.debug("Appending PowerShell vars");
        appendPowerShellVars(connection, scriptBuilder);
        logger.debug("Appending script");
        appendScript(scriptBuilder);
        logger.debug("Replacing all LF's that are NOT preceded by a CR with CRLF.");
        String script = scriptBuilder.toString().replaceAll("(?<!\r)\n", "\r\n");
        logger.debug("Finished generation of PowerShell script.");
        return script;
    }

    private void appendBaseRuntimeScript(StringBuilder b) {
        appendScriptDir(BASE_RUNTIME_PATH, b);
    }

    private void appendContainerRuntimeScripts(StringBuilder b) {
        if(container.hasProperty("runtimePath")) {
            String runtimePath = container.getProperty("runtimePath");
            appendScriptDir(runtimePath, b);
        }
    }

    private void appendContainerLibraryScripts(StringBuilder b) {
        if(container instanceof PowerShellContainer) {
            appendScripts(((PowerShellContainer) container).getLibraryScripts(), b);
        }
    }

    private void appendDeployedLibraryScripts(StringBuilder b) {
        if (deployed instanceof BasePowerShellDeployed) {
            appendScripts(((BasePowerShellDeployed<?, ?>) deployed).getLibraryScripts(), b);
        }
    }

    private void appendPowerShellVars(OverthereConnection connection, StringBuilder b) {
        b.append("# PowerShellVars\n");
        b.append(PowerShellVarsConverter.javaToPowerShell(connection, powerShellVars, uploadArtifactData));
    }

    private void appendScript(StringBuilder b) {
        b.append(loadScript(scriptPath));
    }

    private void doUploadClasspathResources(ExecutionContext ctx, OverthereConnection conn) {
        for (String r : classpathResources) {
            String filename;
            int indexOfLastSlash = r.lastIndexOf('/');
            if(indexOfLastSlash < 0) {
                filename = r;
            } else {
                filename = r.substring(indexOfLastSlash + 1);
            }

            OverthereFile target = conn.getTempFile(filename);
            
            logger.info("Uploading classpath resource [{}] to temporary file [{}].", r, target);
            OutputStream out = target.getOutputStream();
            try {
                URL resourceURL = Resources.getResource(r);
                Resources.copy(resourceURL, out);
            } catch (IOException exc) {
                throw new RuntimeException(exc);
            } finally {
                closeQuietly(out);
            }
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }
    
    public void setClasspathResources(List<String> classpathResources) {
        this.classpathResources = classpathResources;
    }

    private static Logger logger = LoggerFactory.getLogger(PowerShellStep.class);

}
