package com.xebialabs.deployit.plugin.python;

import static com.google.common.collect.Sets.newHashSet;

import java.io.UnsupportedEncodingException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import static java.net.URLDecoder.decode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.inspection.InspectionExecutionContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionStep;
import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;

@SuppressWarnings("serial")
public class PythonInspectionStep extends PythonStep<InspectionExecutionContext> implements InspectionStep {
	private static final String defaultEncoding = "UTF-8";
	
	protected ConfigurationItem inspectedItem;

	protected Descriptor inspectedItemDescriptor;

	public static final String INSPECTED_PROPERTY_PRELUDE = "INSPECTED:";

	public static final String DISCOVERED_ITEM_PRELUDE = "DISCOVERED:";

	public PythonInspectionStep(ConfigurationItem inspectedItem, PythonManagingContainer container, String scriptPath, Map<String, Object> pythonVars, String description) {
		super(container, scriptPath, pythonVars, description);
		this.inspectedItem = inspectedItem;
		this.inspectedItemDescriptor = DescriptorRegistry.getDescriptor(inspectedItem.getType());
	}

	@Override
	public Result execute(final InspectionExecutionContext ctx) throws Exception {
		Result result = doExecute(new InspectionExecutionContext() {
			@Override
			public void logOutput(String output) {
				if (handleOutputLine(output, ctx)) {
					logger.trace("Handled inspection protocol line {}", output);
				} else {
					ctx.logOutput(output);
				}				
			}

			@Override
			public void logError(String error) {
				ctx.logError(error);
			}

			@Override
			public void logError(String error, Throwable t) {
				ctx.logError(error, t);
			}

			@Override
			public Object getAttribute(String name) {
				return ctx.getAttribute(name);
			}

			@Override
			public void setAttribute(String name, Object object) {
				ctx.setAttribute(name, object);
			}

			@Override
			public void discovered(ConfigurationItem item) {
				ctx.discovered(item);
			}
		});
		return result;
	}

	protected boolean handleOutputLine(String line, InspectionExecutionContext ctx) {
		int posOfEqualsSign = line.indexOf('=');
		if (posOfEqualsSign < 0) {
			return false;
		}

		try{
			if (line.startsWith(INSPECTED_PROPERTY_PRELUDE)) {
				String name = decode(line.substring(INSPECTED_PROPERTY_PRELUDE.length(), posOfEqualsSign), defaultEncoding);
				String value = line.substring(posOfEqualsSign + 1);
				return handleInspectedProperty(name, value, ctx);
			}

			if (line.startsWith(DISCOVERED_ITEM_PRELUDE)) {
				String id = decode(line.substring(DISCOVERED_ITEM_PRELUDE.length(), posOfEqualsSign), defaultEncoding);
				String typeName = line.substring(posOfEqualsSign + 1);
				return handleDiscoveredItem(id, typeName, ctx);
			}
		} catch (UnsupportedEncodingException encodingException) {
			logger.debug("Unsupported encoding exception was encountered while parsing inspected and / or discovered items");
			return false;
		}
		return false;
	}

	private boolean handleInspectedProperty(String name, String value, InspectionExecutionContext ctx) throws UnsupportedEncodingException {
	    PropertyDescriptor pd = inspectedItemDescriptor.getPropertyDescriptor(name);
	    if (pd == null) {
	    	ctx.logError("Inspected unknown property " + name);
	    	return false;
	    }

		if (pd.getKind() == PropertyKind.SET_OF_CI) {
			Set<ConfigurationItem> refs = newHashSet();
			Descriptor descriptor = DescriptorRegistry.getDescriptor(pd.getReferencedType());
			for (String id : value.split(",")) {
				ConfigurationItem ref = descriptor.newInstance();
				ref.setId(decode(id,defaultEncoding));
				refs.add(ref);
			}
			pd.set(inspectedItem, refs);
		} else if (pd.getKind() == PropertyKind.SET_OF_STRING){
			Set<String> stringProperties = new HashSet<String>();
			for (String property : value.split(",")) {
				stringProperties.add(decode(property, defaultEncoding));				
			}
			pd.set(inspectedItem, stringProperties);
		} else {	
			pd.set(inspectedItem, decode(value, defaultEncoding));
		}

	    logger.debug("Inspected property {} with value {}", name, value);
	    return true;
    }

	private boolean handleDiscoveredItem(String id, String typeName, InspectionExecutionContext ctx) throws UnsupportedEncodingException {
	    Type type = Type.valueOf(decode(typeName, defaultEncoding));
	    if (!DescriptorRegistry.exists(type)) {
	    	ctx.logError("Discovered item " + id + " of unknown type " + type);
	    	return false;
	    }
	    Descriptor d = DescriptorRegistry.getDescriptor(type);
	    ConfigurationItem discoveredItem = d.newInstance();
	    discoveredItem.setId(id);
	    
	    for(PropertyDescriptor pd : d.getPropertyDescriptors()) {
	    	if(isReferenceToParent(pd)) {
	    		pd.set(discoveredItem, inspectedItem);
	    	}
	    }

	    ctx.discovered(discoveredItem);
	    logger.debug("Discovered item {} with type {}", id, type);
	    return true;
    }

	protected boolean isReferenceToParent(PropertyDescriptor pd) {
	    return pd.getKind() == PropertyKind.CI && pd.isAsContainment() && inspectedItemDescriptor.isAssignableTo(pd.getReferencedType());
    }

	private static Logger logger = LoggerFactory.getLogger(PythonInspectionStep.class);

}
