package com.xebialabs.deployit.plugin.python;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.InterruptedIOException;
import java.io.OutputStream;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener;
import com.xebialabs.deployit.engine.spi.execution.StepExecutionStateEvent;
import com.xebialabs.deployit.engine.spi.execution.TaskExecutionStateEvent;
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcess;
import com.xebialabs.overthere.RuntimeIOException;

import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.STOPPED;

import static com.xebialabs.deployit.engine.api.execution.TaskExecutionState.EXECUTED;

import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.CONNECT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.DISCONNECT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.RUN_SCRIPT_FROM_DAEMON;
import static com.xebialabs.deployit.plugin.python.PythonStep.STANDARD_RUNTIME_PATH;
import static com.xebialabs.deployit.plugin.python.PythonStep.dumpPythonScript;
import static com.xebialabs.deployit.plugin.python.PythonVarsConverter.toPythonString;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;

@SuppressWarnings("serial")
class PythonDaemon implements ExecutionStateListener {

    private static final String DAEMON_SCRIPT = "python/daemon/daemon.py";
    private static final String DAEMON_STARTED = "DEPLOYIT-DAEMON-STARTED";
    private static final String DAEMON_EXIT_CODE_MARKER = "DEPLOYIT-DAEMON-EXIT-VALUE: ";
    private static final String DAEMON_END_OF_STDOUT_MARKER = "DEPLOYIT-DAEMON-END-OF-STDOUT";
    private static final String DAEMON_END_OF_STDERR_MARKER = "DEPLOYIT-DAEMON-END-OF-STDERR";
    private static final int FLUSH_DELAY_MS = 5000;
    private static final int FLUSH_CHECK_INTERVAL_MS = 2000;

    private static final Timer flushTimer = new Timer("PythonDaemon-AutoFlushTimer", true);

    private final PythonManagingContainer container;

    private transient OverthereConnection connection;

    private transient OverthereProcess process;

    PythonDaemon(final PythonManagingContainer container) {
        this.container = container;
    }

    boolean isAlive() {
        return connection != null && process != null;
    }

    void start(ExecutionContext context) {
        context.logOutput("Starting Python daemon on " + container.getHost());

        this.connection = container.getHost().getConnection();
        logger.info("Starting Python daemon on {}", connection);

        OverthereFile uploadedDaemonScript = uploadDaemonScript();

        waitForDaemonStart(context, uploadedDaemonScript);
    }

    private OverthereFile uploadDaemonScript() {
        String daemonScript = generateDaemonScript();
        dumpPythonScript("daemon.py", daemonScript);
        return uploadScript(connection, "daemon.py", daemonScript);
    }

    private String generateDaemonScript() {
        StringBuilder b = generateStandardRuntimeScript(container);
        b.append("#\n" + CONNECT_FROM_DAEMON + "()\n");
        b.append(loadScript(DAEMON_SCRIPT));
        b.append("#\n" + DISCONNECT_FROM_DAEMON + "()\n");
        return b.toString();
    }

    static StringBuilder generateStandardRuntimeScript(PythonManagingContainer c) {
        StringBuilder b = new StringBuilder();
        b.append(loadScriptDir(STANDARD_RUNTIME_PATH));
        //for backward compatibility
        b.append(loadScriptDir(c.getRuntimePath()));
        //for synthetic extensibility
        if (c.hasProperty("libraryScripts")) {
            List<String> scripts = c.getProperty("libraryScripts");
            for(String script: scripts) {
                b.append(loadScript(script));
            }
        }
        return b;
    }

    private void waitForDaemonStart(final ExecutionContext context, final OverthereFile uploadedDaemonScript) {
        process = connection.startProcess(container.getScriptCommandLine(uploadedDaemonScript));
        final InputStreamReader stdout = new InputStreamReader(process.getStdout());
        final StringBuilder stdoutLineBuffer = new StringBuilder();
        final long[] flushAfter = new long[1];
        final TimerTask flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                synchronized (stdoutLineBuffer) {
                    if (flushAfter[0] < System.currentTimeMillis()) {
                        if (stdoutLineBuffer.length() > 0) {
                            context.logOutput(stdoutLineBuffer.toString());
                            stdoutLineBuffer.setLength(0);
                        }
                        flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                    }
                }
            }
        };
        flushTimer.schedule(flushTimerTask, FLUSH_DELAY_MS, FLUSH_CHECK_INTERVAL_MS);
        try {
            for (;;) {
                try {
                    int cInt = stdout.read();
                    if (cInt == -1) {
                        captureStderr(context, process);
                        int exitCode = -1;
                        try {
                            exitCode = process.waitFor();
                        } catch (InterruptedException exc) {
                            logger.error("Interrupted while waiting for " + process + " to complete");
                            Thread.currentThread().interrupt();
                        }
                        throw new PythonDaemonException("Cannot start python daemon: exit code " + exitCode);
                    } else {
                        char c = (char) cInt;
                        if (c != '\r' && c != '\n') {
                            synchronized (stdoutLineBuffer) {
                                stdoutLineBuffer.append(c);
                            }
                        }
                        if (c == '\n') {
                            synchronized (stdoutLineBuffer) {
                                flushAfter[0] = System.currentTimeMillis() + FLUSH_DELAY_MS;
                                String stdoutLine = stdoutLineBuffer.toString();
                                context.logOutput(stdoutLine);
                                stdoutLineBuffer.setLength(0);
                                if (stdoutLine.startsWith(DAEMON_STARTED)) {
                                    break;
                                }
                            }
                        }
                    }
                } catch (IOException exc) {
                    throw new PythonDaemonException("Cannot start python daemon", exc);
                }
            }
        } finally {
            flushTimerTask.cancel();
        }
    }

    private static void captureStderr(final ExecutionContext context, OverthereProcess daemonProcess) throws IOException {
        final BufferedReader stderr = new BufferedReader(new InputStreamReader(daemonProcess.getStderr()));
        for (;;) {
            final String stderrLine = stderr.readLine();
            if (stderrLine == null) {
                break;
            }
            context.logError(stderrLine);
        }
    }

    int executePythonScript(final ExecutionContext ctx, final OverthereFile script) {
        try {
            logger.info("Executing Python script {} on {} (with daemon)", script, connection);
            String daemonLine = RUN_SCRIPT_FROM_DAEMON + "(" + toPythonString(script.getPath()) + ")\n";
            OutputStream stdin = process.getStdin();
            stdin.write(daemonLine.getBytes());
            stdin.flush();

            final BufferedReader stdout = new BufferedReader(new InputStreamReader(process.getStdout()));
            final BufferedReader stderr = new BufferedReader(new InputStreamReader(process.getStderr()));
            Thread stderrThread = new Thread(new Runnable() {
                public void run() {
                    try {
                        while (!Thread.interrupted()) {
                            String stderrLine = stderr.readLine();
                            if (stderrLine == null || stderrLine.equals(DAEMON_END_OF_STDERR_MARKER)) {
                                break;
                            }
                            ctx.logError(stderrLine);
                        }
                    } catch(InterruptedIOException ie) {
                        Thread.currentThread().interrupt();
                    } catch (IOException exc) {
                        exc.printStackTrace();
                    }
                }
            }, "stderr printer for python daemon running on " + this);
            stderrThread.start();

            int exitCode = 0;
            for (;;) {
                String stdoutLine = stdout.readLine();
                if (stdoutLine == null) {
                    throw new RuntimeIOException("Cannot execute script " + script.getPath() + " on " + container.getHost() + ": lost connection to the python daemon");
                }

                if (stdoutLine.startsWith(DAEMON_EXIT_CODE_MARKER)) {
                    try {
                        exitCode = Integer.parseInt(stdoutLine.substring(DAEMON_EXIT_CODE_MARKER.length()));
                    } catch (NumberFormatException ignored) {
                    }
                } else if (stdoutLine.equals(DAEMON_END_OF_STDOUT_MARKER)) {
                    break;
                } else if (stdoutLine.equals(DAEMON_END_OF_STDERR_MARKER)) {
                    //Using sudo causes the end of stderr marker to be sent to stdout instead of stderr.
                    //Terminate the error thread, otherwise the daemon will hang.
                    stderrThread.interrupt();
                } else {
                    ctx.logOutput(stdoutLine);
                }
            }
            try {
                stderrThread.join();
            } catch (InterruptedException exc) {
                logger.error("Interrupted while waiting for " + stderrThread + " to complete");
                Thread.currentThread().interrupt();
            }
            return exitCode;
        } catch (IOException exc) {
            throw new RuntimeIOException("Cannot execute script " + script.getPath() + " on " + container.getHost(), exc);
        }
    }

    private void quit() {
        logger.info("Stopping Python daemon on {}", connection);
        OutputStream stdin = process.getStdin();
        try {
            stdin.write("QUIT\r\n".getBytes());
            stdin.flush();
            process.waitFor();
            process = null;
        } catch (IOException exc) {
            logger.error("Error stopping python daemon", exc);
        } catch (RuntimeException exc) {
            logger.error("Error stopping python daemon", exc);
        } catch (InterruptedException exc) {
            logger.error("Error stopping python daemon", exc);
        } finally {
            Closeables.closeQuietly(connection);
            connection = null;
        }
    }

    @Override
    public void stepStateChanged(StepExecutionStateEvent event) {
        // Do nothing
    }

    @Override
    public void taskStateChanged(TaskExecutionStateEvent event) {
        if (isAlive() && (event.currentState().isFinal() || event.currentState() == EXECUTED || event.currentState() == STOPPED)) {
            quit();
        }
    }

    private static final Logger logger = LoggerFactory.getLogger(PythonDaemon.class);

}
