package com.xebialabs.deployit.plugin.python;

import static com.google.common.collect.Lists.newArrayList;
import static org.apache.commons.codec.binary.Base64.encodeBase64String;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.Collections2;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.overthere.OverthereConnection;

public class PythonVarsConverter extends VarsConverter {

    public static String javaToPython(OverthereConnection connection, Map<String, Object> pythonVars) {
        return javaToPython(connection, pythonVars, true);
    }

    public static String javaToPython(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        List<String> pythonList = javaToPythonList(connection, pythonVars,uploadArtifactData);
        return Joiner.on("\n").join(pythonList) + "\n";
    }

    static List<String> javaToPythonList(OverthereConnection connection, Map<String, Object> pythonVars, boolean uploadArtifactData) {
        PythonVarsConverter converter = new PythonVarsConverter(connection, pythonVars);
        converter.setUploadArtifactData(uploadArtifactData);
        return converter.convert();
    }

    private PythonVarsConverter(OverthereConnection connection, Map<String, Object> pythonVars) {
        super(connection, pythonVars, DerivedProperty.class);
    }

    protected void setNullVariable(String varName) {
        add(varName + " = None");
    }

    protected void setBooleanVariable(String varName, boolean varValue) {
        add(varName + " = " + (varValue ? "True" : "False"));
        add(varName + "_as_string = " + (varValue ? "\'true\'" : "'false\'"));
    }

    protected void setIntegerVariable(String varName, int varValue) {
        add(varName + " = " + varValue);
    }

    protected void setLongVariable(String varName, long varValue) {
        add(varName + " = " + varValue + "L");
    }

    protected void setStringVariable(String varName, String varValue) {
        add(varName + " = " + toPythonString(varValue));
    }

    protected void setCiReferenceVariable(String varName, ConfigurationItem varValue) {
        add(varName + " = " + getConfigurationItemVariableName(varValue));
    }

    protected void setEmptyCollectionVariable(String varName) {
        add(varName + " = []");
    }

    protected void setCollectionOfStringsVariable(String varName, Collection<?> varValue) {
        if (varValue.isEmpty()) {
            setEmptyCollectionVariable(varName);
        } else {
            Collection<String> encodedStrings = Collections2.transform(varValue, new Function<Object, String>() {
                public String apply(Object input) {
                    return toPythonString(input.toString());
                }
            });
            add(varName + " = [" + Joiner.on(", ").join(encodedStrings) + "]");
        }
    }

    protected void createObject(String varName) {
        add(varName + " = DictionaryObject()");
    }

    protected void setNullProperty(String objectName, String propertyName) {
        setNullVariable(objectName + "." + propertyName);
    }

    protected void setEmptyCollectionProperty(String objectName, String propertyName) {
        setEmptyCollectionVariable(objectName + "." + propertyName);
    }

    protected void setEmptyMapProperty(String objectName, String propertyName) {
        add(objectName + "." + propertyName + " = {}");
    }

    protected void setBooleanProperty(String objectName, String propertyName, boolean varValue) {
        setBooleanVariable(objectName + "." + propertyName, varValue);
    }

    protected void setIntegerProperty(String objectName, String propertyName, int varValue) {
        setIntegerVariable(objectName + "." + propertyName, varValue);
    }

    protected void setStringProperty(String objectName, String propertyName, String varValue) {
        setStringVariable(objectName + "." + propertyName, varValue);
    }

    protected void setPasswordProperty(String objectName, String propertyName, String varValue) {
        add(objectName + "." + propertyName + " = base64.decodestring(" + toPythonString(encodeBase64String(varValue.getBytes())) + ")");
    }

    protected void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> varValue) {
        setCollectionOfStringsVariable(objectName + "." + propertyName, varValue);
    }

    protected void setCiReferenceProperty(String objectName, String propertyName, ConfigurationItem varValue) {
        add(objectName + "." + propertyName + " = " + getConfigurationItemVariableName(varValue));
    }

    protected void setCollectionOfCiReferencesProperty(String objectName, String propertyName, Collection<ConfigurationItem> varValue) {
        if (varValue.isEmpty()) {
            setEmptyCollectionVariable(objectName + "." + propertyName);
        } else {
            List<String> varRefs = newArrayList();
            for (ConfigurationItem setItem : varValue) {
                varRefs.add(getConfigurationItemVariableName(setItem));
            }
            add(objectName + "." + propertyName + " = [" + Joiner.on(", ").join(varRefs) + "]");
        }
    }

    protected void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> varValue) {
        if (varValue.isEmpty()) {
            setEmptyMapProperty(objectName, propertyName);
        } else {
            List<String> entries = newArrayList();
            for (String key : varValue.keySet()) {
                String value = varValue.get(key);
                if (value == null) {
                    entries.add(toPythonString(key) + ": None");
                } else {
                    entries.add(toPythonString(key) + ": " + toPythonString(value));
                }
            }
            add(objectName + "." + propertyName + " = {" + Joiner.on(", ").join(entries) + "}");
        }
    }

    protected void createObjectAndSetObjectProperty(String objectName, String propertyName) {
        createObject(objectName + "." + propertyName);
    }

    public static String toPythonString(String str) {
        StringBuilder converted = new StringBuilder();
        converted.append("\"");
        for (int i = 0; i < str.length(); i++) {
            char c = str.charAt(i);
            switch (c) {
            case '\r':
                converted.append("\\r");
                break;
            case '\n':
                converted.append("\\n");
                break;
            case '\\':
            case '\'':
            case '\"':
                converted.append('\\');
                // deliberate fall-through
            default:
                converted.append(c);
            }
        }
        converted.append("\"");
        return converted.toString();
    }

}
