package com.xebialabs.deployit.plugin.python;

import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.io.Closeables.closeQuietly;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.CONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.DISCONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

@SuppressWarnings("serial")
abstract class PythonStep implements Step {

    static final String STANDARD_RUNTIME_PATH = "python/runtime";

    private final PythonManagingContainer container;
    private final String scriptPath;
    private final Map<String, Object> pythonVars;
    private final String description;
    private boolean uploadArtifactData = true;

    public PythonStep(PythonManagingContainer container, String scriptPath, Map<String, Object> pythonVars, String description) {
        this.container = checkNotNull(container, "container is null");
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.pythonVars = checkNotNull(pythonVars, "pythonVars is null");
        this.description = checkNotNull(description, "description is null");
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
        try {
            OverthereConnection conn = container.getHost().getConnection();
            try {
                String finalScript = aggregateScript(conn);
                dumpPythonScript(getName(scriptPath), finalScript);
                OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, finalScript);

                int res = executePythonScript(ctx, conn, uploadedScriptFile);

                logger.debug("Exit code: {}", res);
                if(res == 0) {
                    return StepExitCode.SUCCESS;
                } else {
                    return StepExitCode.FAIL;
                }
            } finally {
                Closeables.closeQuietly(conn);
            }
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    protected String aggregateScript(OverthereConnection connection) {
        String pythonVarsPython = PythonVarsConverter.javaToPython(connection, pythonVars, uploadArtifactData);
        StringBuilder b = new StringBuilder();
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b = PythonDaemon.generateStandardRuntimeScript(container);
        }

        //for synthetic extensibility
        PythonManagedDeployed<?,?> deployed = (PythonManagedDeployed<?, ?>) pythonVars.get("deployed");
        deployed = deployed == null ? (PythonManagedDeployed<?, ?>) pythonVars.get("prototype") : deployed;
        if (deployed != null) {
            for(String script: deployed.getLibraryScripts()) {
                b.append(loadScript(script));
            }
        }

        b.append("# PythonVars\n");
        b.append(pythonVarsPython);
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b.append("#\n" + CONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
        }
        b.append(loadScript(scriptPath));
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b.append("#\n" + DISCONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
        }
        return b.toString();
    }

    private int executePythonScript(final ExecutionContext ctx, final OverthereConnection conn, final OverthereFile script) {
        if (container.runWithDaemon() && conn.canStartProcess()) {
            return getDaemon(ctx).executePythonScript(ctx, script);
        } else {
            return executePythonScriptDirectly(ctx, conn, script);
        }
    }

    protected int executePythonScriptDirectly(final ExecutionContext ctx, final OverthereConnection conn, final OverthereFile script) {
        logger.info("Executing Python script {} on {} (without daemon)", script, conn);
        DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
        DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx);
        try {
            return conn.execute(stdoutHandler, stderrHandler, container.getScriptCommandLine(script));
        } finally {
            closeQuietly(stdoutHandler);
            closeQuietly(stderrHandler);
        }
    }

    private PythonDaemon getDaemon(final ExecutionContext context) {
        String key = "DAEMON_" + container.getId();
        PythonDaemon daemon = (PythonDaemon) context.getAttribute(key);
        if (daemon == null || !daemon.isAlive()) {
            daemon = new PythonDaemon(container);
            daemon.start(context);
            context.setAttribute(key, daemon);
        }
        return daemon;
    }

    String getScriptPath() {
        return scriptPath;
    }

    Map<String, Object> getPythonVars() {
        return pythonVars;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String toString() {
        return scriptPath;
    }

    public static void dumpPythonScript(String scriptName, String scriptContents) {
        dumpScript(scriptName, scriptContents, scriptsLogger);
    }

    private static final Logger logger = LoggerFactory.getLogger(PythonStep.class);

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.python.scripts");

}
