package com.xebialabs.deployit.plugin.python;

import java.util.List;
import java.util.Map;
import com.google.common.base.Strings;

import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Operation;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseDeployed;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@SuppressWarnings("serial")
public abstract class PythonManagedDeployed<D extends Deployable, C extends PythonManagedContainer> extends BaseDeployed<D, C> {

    @Property(required = false, hidden = true, description = "List of scripts to appended to the the deployeds runtime script")
    private List<String> libraryScripts = newArrayList();

    @Property(hidden = true, defaultValue = "50", description = "The order in which a discover step will be executed.")
    private int discoverOrder;

    @Property(required = false, hidden = true, defaultValue = "false", description = "flag to indicate whether the deployed application CI is to be injected to the python script execution context.")
    private boolean exposeDeployedApplication;

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String scriptPath, String verb) {
        return addStep(ctx, order, scriptPath, verb, true, null);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String scriptPath, String verb, boolean uploadArtifactData) {
        return addStep(ctx, order, scriptPath, verb, uploadArtifactData, null);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String scriptPath, String verb, CheckpointInfo checkpoint) {
        return addStep(ctx, order, scriptPath, verb, true, checkpoint);
    }

    protected boolean addStep(DeploymentPlanningContext ctx, int order, String scriptPath, String verb, boolean uploadArtifactData, CheckpointInfo checkpoint) {
        if (!Strings.isNullOrEmpty(scriptPath)) {
            PythonDeploymentStep step = new PythonDeploymentStep(order, getContainer().getManagingContainer(), scriptPath, getPythonVars(ctx.getDeployedApplication()), getDescription(verb));
            step.setUploadArtifactData(uploadArtifactData);
            if (checkpoint == null) {
                ctx.addStep(step);
            } else {
                ctx.addStepWithCheckpoint(step, checkpoint.delta, checkpoint.operation);
            }
            return true;
        } else {
            return false;
        }
    }

    protected boolean addStep(InspectionContext ctx, String scriptPath, String verb) {
        if (!Strings.isNullOrEmpty(scriptPath)) {
            ctx.addStep(new PythonInspectionStep(this, getContainer().getManagingContainer(), scriptPath, getPythonVars(null), getDescription(verb)));
            return true;
        } else {
            return false;
        }
    }

    protected Map<String, Object> getPythonVars() {
        return getPythonVars(null);
    }

    protected Map<String, Object> getPythonVars(DeployedApplication deployedApplication) {
        Map<String, Object> pythonVars = newHashMap();
        pythonVars.put("deployed", this);
        if(exposeDeployedApplication) {
            checkNotNull(deployedApplication, "deployedApplication is null");
            pythonVars.put("deployedApplication", deployedApplication);
        }
        return pythonVars;
    }

    protected String getDescription(String verb) {
        return String.format("%s %s on %s", verb, getName(), getContainer().getName());
    }

    public boolean getExposeDeployedApplication() {
        return exposeDeployedApplication;
    }

    public void setExposeDeployedApplication(boolean exposeDeployedApplication) {
        this.exposeDeployedApplication = exposeDeployedApplication;
    }

    public List<String> getLibraryScripts() {
        return libraryScripts;
    }

    public int getDiscoverOrder() {
        return discoverOrder;
    }

    public void setDiscoverOrder(int discoverOrder) {
        this.discoverOrder = discoverOrder;
    }

    protected static CheckpointInfo checkpoint(Delta delta) {
        return new CheckpointInfo(delta);
    }

    protected static CheckpointInfo checkpoint(Delta delta, Operation operation) {
        return new CheckpointInfo(delta, operation);
    }

    protected static class CheckpointInfo {
        public final Delta delta;
        public final Operation operation;

        CheckpointInfo(Delta delta) {
            this.delta = delta;
            this.operation = delta.getOperation();
        }

        CheckpointInfo(Delta delta, Operation operation) {
            this.delta = delta;
            this.operation = operation;
        }
    }

    public List<Step> controlTaskDispatch(String name, Map<String,String> args) {
        return ControlTaskDelegate.dispatch(name, args, this, getContainer());
    }
 }

