package com.xebialabs.deployit.plugin.python;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Preview;
import com.xebialabs.deployit.plugin.api.flow.PreviewStep;
import com.xebialabs.deployit.plugin.api.flow.StageableStep;
import com.xebialabs.deployit.plugin.api.flow.StagedFile;
import com.xebialabs.deployit.plugin.api.flow.StagingContext;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.remoting.vars.VarsConverter;
import com.xebialabs.deployit.plugin.steps.CalculatedStep;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.CONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.python.PythonManagingContainer.DISCONNECT_FROM_STAND_ALONE_SCRIPT;
import static com.xebialabs.deployit.plugin.remoting.preview.PreviewOverthereConnection.getPreviewConnection;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.MDC_KEY_SCRIPT_PATH;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScriptDir;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.appendScripts;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.dumpScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.loadScript;
import static com.xebialabs.deployit.plugin.remoting.scripts.ScriptUtils.uploadScript;
import static com.xebialabs.overthere.util.OverthereUtils.getName;

@SuppressWarnings("serial")
abstract class PythonStep extends CalculatedStep implements PreviewStep, StageableStep {

    private static final String BASE_RUNTIME_PATH = "python/runtime";

    private PythonManagingContainer container;

    private String scriptPath;

    private Map<String, Object> scriptContext = new HashMap<>();

    private boolean uploadArtifactData = true;

    private Preview preview;

    public PythonStep(PythonManagingContainer container, String scriptPath, Map<String, Object> scriptContext, Integer order, String description) {
        super(checkNotNull(order, "order is null"), checkNotNull(description, "description is null"));
        this.container = checkNotNull(container, "container is null");
        this.scriptPath = checkNotNull(scriptPath, "scriptPath is null");
        this.scriptContext = checkNotNull(scriptContext, "scriptContext is null");
    }

    @SuppressWarnings("unused")
    public PythonStep(PythonManagingContainer container, String scriptPath, Map<String, Object> scriptContext, String description) {
        this(container, scriptPath, scriptContext, 50, description);
    }

    @Override
    public void requestStaging(StagingContext ctx) {
        if (!uploadArtifactData) {
            return;
        }

        Map<String, Object> stagedFilesMap = new HashMap<>(getPythonVars());
        for (Map.Entry<String, Object> entry : scriptContext.entrySet()) {
            if (entry.getValue() instanceof Artifact) {
                StagedFile stagedFile = ctx.stageArtifact((Artifact) entry.getValue(), container.getHost());
                stagedFilesMap.put(entry.getKey() + VarsConverter.STAGED_FILE_VARIABLE_NAME_PREFIX, stagedFile);
            }
        }
        scriptContext = stagedFilesMap;
    }

    protected StepExitCode doExecute(ExecutionContext ctx) throws Exception {
        MDC.put(MDC_KEY_SCRIPT_PATH, scriptPath);
        try {
            try(OverthereConnection conn = container.getHost().getConnection()) {
                String generatedScript = aggregateScript(conn, ctx, false);
                String obfuscatedScript = aggregateScript(conn, ctx, true);
                dumpPythonScript(getName(scriptPath), obfuscatedScript);
                OverthereFile uploadedScriptFile = uploadScript(conn, scriptPath, generatedScript);

                int res = executePythonScript(ctx, conn, uploadedScriptFile);

                logger.debug("Exit code: {}", res);
                if(res == 0) {
                    return StepExitCode.SUCCESS;
                } else {
                    return StepExitCode.FAIL;
                }
            }
        } finally {
            MDC.remove(MDC_KEY_SCRIPT_PATH);
        }
    }

    @Override
    public Preview getPreview() {
        if(preview == null) {
            try (OverthereConnection poc = getPreviewConnection()) {
                String contents = aggregateScript(poc, null, true);
                preview = Preview.withSourcePathAndContents(scriptPath, contents);
            }
        }
        return preview;
    }

    protected String aggregateScript(OverthereConnection connection, ExecutionContext ctx, boolean maskPasswords) {
        String pythonVarsPython = PythonVarsConverter.javaToPython(connection, scriptContext, uploadArtifactData, ctx, maskPasswords);
        StringBuilder b = new StringBuilder();

        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            appendRuntimeScripts(container, b);
        }

        appendDeployedRuntimeScripts(b);

        b.append("# PythonVars\n");
        b.append(pythonVarsPython);
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b.append("#\n" + CONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
        }
        b.append(loadScript(scriptPath));
        if(!container.runWithDaemon() || !connection.canStartProcess()) {
            b.append("#\n" + DISCONNECT_FROM_STAND_ALONE_SCRIPT + "()\n");
        }
        return b.toString();
    }

    static void appendRuntimeScripts(PythonManagingContainer c, StringBuilder b) {
        appendBaseRuntimeScripts(b);
        appendContainerRuntimeScripts(c, b);
        appendContainerLibraryScripts(c, b);
    }

    private static void appendBaseRuntimeScripts(StringBuilder b) {
        appendScriptDir(BASE_RUNTIME_PATH, b);
    }

    private static void appendContainerRuntimeScripts(PythonManagingContainer c, StringBuilder b) {
        appendScriptDir(c.getRuntimePath(), b);
    }

    private static void appendContainerLibraryScripts(PythonManagingContainer c, StringBuilder b) {
        if (c.hasProperty("libraryScripts")) {
            List<String> scripts = c.getProperty("libraryScripts");
            appendScripts(scripts, b);
        }
    }

    private void appendDeployedRuntimeScripts(StringBuilder b) {
        PythonManagedDeployed<?,?> deployed = (PythonManagedDeployed<?, ?>) scriptContext.get("deployed");
        deployed = deployed == null ? (PythonManagedDeployed<?, ?>) scriptContext.get("prototype") : deployed;
        if (deployed != null) {
            for(String script: deployed.getLibraryScripts()) {
                b.append(loadScript(script));
            }
        }
    }

    private int executePythonScript(final ExecutionContext ctx, final OverthereConnection conn, final OverthereFile script) {
        if (container.runWithDaemon() && conn.canStartProcess()) {
            return getDaemon(ctx).executePythonScript(ctx, script);
        } else {
            return executePythonScriptDirectly(ctx, conn, script);
        }
    }

    protected int executePythonScriptDirectly(final ExecutionContext ctx, final OverthereConnection conn, final OverthereFile script) {
        logger.info("Executing Python script {} on {} (without daemon)", script, conn);
        try (DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
             DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx)
        ) {
            return conn.execute(stdoutHandler, stderrHandler, container.getScriptCommandLine(script));
        }
    }

    private PythonDaemon getDaemon(final ExecutionContext context) {
        String key = "DAEMON_" + container.getId();
        PythonDaemon daemon = (PythonDaemon) context.getAttribute(key);
        if (daemon == null || !daemon.isAlive()) {
            daemon = new PythonDaemon(container);
            context.setAttribute(key, daemon);
            daemon.start(context);
        }
        return daemon;
    }

    String getScriptPath() {
        return scriptPath;
    }

    Map<String, Object> getPythonVars() {
        return scriptContext;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    @Override
    public String toString() {
        return scriptPath;
    }

    public static void dumpPythonScript(String scriptName, String scriptContents) {
        dumpScript(scriptName, scriptContents, scriptsLogger);
    }

    private static final Logger logger = LoggerFactory.getLogger(PythonStep.class);

    private static final Logger scriptsLogger = LoggerFactory.getLogger("com.xebialabs.deployit.plugin.python.scripts");

}
