package com.xebialabs.deployit.plugin.overthere;

import java.io.Closeable;
import java.util.Timer;
import java.util.TimerTask;

import com.xebialabs.overthere.OverthereProcessOutputHandler;

/**
 * A {@link OverthereProcessOutputHandler} that delegates actual logging to its subclass. When no output has
 * been sent to {@code stdout} in a while, it is flushed automatically.
 */
public abstract class AutoFlushingProcessOutputHandler implements OverthereProcessOutputHandler, Closeable {

    public static final int FLUSH_DELAY_MS = 5000;
    public static final int FLUSH_CHECK_INTERVAL_MS = 2000;

    private static final Timer flushTimer = new Timer("AutoFlushTimer", true);

    private StringBuilder lineBuffer;
    private long flushAfter;
    private TimerTask flushTimerTask;

    public AutoFlushingProcessOutputHandler() {
        this.lineBuffer = new StringBuilder();
        this.flushAfter = nextFlushTime();
        this.flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                checkFlushNeeded();
            }
        };
        flushTimer.schedule(this.flushTimerTask, FLUSH_DELAY_MS, FLUSH_CHECK_INTERVAL_MS);
    }

    public final void handleOutputLine(String line) {
        // no-op
    }

    public final void handleOutput(char c) {
        if (c != '\r' && c != '\n') {
            appendToLineBuffer(c);
        }
        if (c == '\n') {
            flushLineBuffer();
        }
    }

    private synchronized void appendToLineBuffer(char c) {
        lineBuffer.append(c);
    }

    private synchronized void checkFlushNeeded() {
        if(flushAfter < System.currentTimeMillis()) {
            flushLineBuffer();
        }
    }

    private synchronized void flushLineBuffer() {
        if (lineBuffer.length() > 0) {
            handleOutput(lineBuffer.toString());
            lineBuffer.setLength(0);
        }
        flushAfter = nextFlushTime();
    }

    protected abstract void handleOutput(String buffer);

    private static long nextFlushTime() {
        return System.currentTimeMillis() + FLUSH_DELAY_MS;
    }

    @Override
    public synchronized void close() {
        flushTimerTask.cancel();
    }

}
