/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */
package com.xebialabs.deployit.plugin.overthere.step;

import com.google.common.io.Closeables;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultProcessOutputHandler;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;
import com.xebialabs.overthere.OverthereProcessOutputHandler;
import com.xebialabs.overthere.util.OverthereUtils;

import static com.google.common.base.Throwables.getStackTraceAsString;
import static com.xebialabs.overthere.ConnectionOptions.ADDRESS;
import static com.xebialabs.overthere.OperatingSystemFamily.UNIX;
import static java.lang.String.format;

@SuppressWarnings("serial")
public class CheckConnectionStep implements Step {

    private Host host;

    public CheckConnectionStep(Host host) {
        this.host = host;
    }

    @Override
    public String getDescription() {
        return "Check connection to " + host.getId();
    }

    @Override
    public int getOrder() {
        return 50;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        DefaultProcessOutputHandler handler = new DefaultProcessOutputHandler(ctx);
        try {
            String hostName = host.getName();
            if (host.hasProperty(ADDRESS)) {
                hostName = host.getProperty(ADDRESS);
            }
            try {
                OverthereConnection connection = host.getConnection();
                try {
                    ctx.logOutput("Attempting to list the contents of the temporary directory");
                    ctx.logOutput("----------------------------------------------------------");
                    echoTmpDirContents(connection, handler);
                    ctx.logOutput("----------------------------------------------------------");
                    ctx.logOutput("Uploading file to the temporary directory");
                    ctx.logOutput("----------------------------------------------------------");
                    OverthereFile fileToUpload = uploadFile(connection);
                    ctx.logOutput("-----------------------------------------");
                    ctx.logOutput("Checking read permission of uploaded file");
                    ctx.logOutput("-----------------------------------------");
                    tryCanGetFilePermissions(connection, fileToUpload);
                    ctx.logOutput("-----------------------------------------");
                } finally {
                    Closeables.closeQuietly(connection);
                }
            } catch (Exception exc) {

                handler.handleErrorLine(format("Connection test to host " + hostName + " failed. Please check address, username and password. Exception message is: %s%n%s",
                        exc, getStackTraceAsString(exc)));
                return StepExitCode.FAIL;
            }
            handler.handleOutputLine("Connection test to host " + hostName + " succeeded");
            return StepExitCode.SUCCESS;
        } finally {
            handler.close();
        }
    }

    private void echoTmpDirContents(OverthereConnection connection, OverthereProcessOutputHandler handler) {
        CmdLine cmdLine = new CmdLine();
        if (connection.getHostOperatingSystem() == UNIX) {
            cmdLine.addArgument("ls");
        } else {
            cmdLine.addArgument("cmd");
            cmdLine.addArgument("/c");
            cmdLine.addArgument("dir");
        }

        String tempDir = host.getTemporaryDirectoryPath();
        if (tempDir == null) {
            tempDir = connection.getHostOperatingSystem().getDefaultTemporaryDirectoryPath();
        }

        cmdLine.addArgument(tempDir);
        connection.execute(handler, cmdLine);
    }

    private static OverthereFile uploadFile(OverthereConnection connection) {
        OverthereFile fileToUpload = connection.getTempFile("hostconnection-remote", ".txt");
        OverthereUtils.write("Contents of host connection test file", "UTF-8", fileToUpload);
        return fileToUpload;
    }

    private static void tryCanGetFilePermissions(OverthereConnection connection, OverthereFile uploadedFile) {
        OverthereFile filesJustUploadedNotAsATempFile = connection.getFile(uploadedFile.getPath());
        if (!filesJustUploadedNotAsATempFile.canRead()) {
            throw new RuntimeException("Cannot check read permission for file just uploaded");
        }
    }
}