package com.xebialabs.deployit.plugin.overthere;

import java.io.Closeable;
import java.util.Timer;
import java.util.TimerTask;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.overthere.OverthereExecutionOutputHandler;

/**
 * An {@link OverthereExecutionOutputHandler} that delegates actual logging to its subclass. When no output has been
 * sent to the stream in a while, it is flushed automatically.
 */
public class DefaultExecutionOutputHandler implements OverthereExecutionOutputHandler, Closeable {

    public static final int FLUSH_DELAY_MS = 5000;
    public static final int FLUSH_CHECK_INTERVAL_MS = 2000;

    private static final Timer flushTimer = new Timer("AutoFlushTimer", true);

    private ExecutionContext context;
    private boolean stdout;
    private StringBuilder lineBuffer;
    private long flushAfter;
    private TimerTask flushTimerTask;

    DefaultExecutionOutputHandler(ExecutionContext context, boolean stdout) {
        this.context = context;
        this.stdout = stdout;
        this.lineBuffer = new StringBuilder();
        this.flushAfter = nextFlushTime();
        this.flushTimerTask = new TimerTask() {
            @Override
            public void run() {
                checkFlushNeeded();
            }
        };
        flushTimer.schedule(this.flushTimerTask, FLUSH_DELAY_MS, FLUSH_CHECK_INTERVAL_MS);
    }

    private static long nextFlushTime() {
        return System.currentTimeMillis() + FLUSH_DELAY_MS;
    }

    private synchronized void checkFlushNeeded() {
        if (flushAfter < System.currentTimeMillis()) {
            flushLineBuffer();
        }
    }

    @Override
    public final void handleLine(String line) {
        // no-op
    }

    @Override
    public final void handleChar(char c) {
        if (c != '\r' && c != '\n') {
            appendToLineBuffer(c);
        }
        if (c == '\n') {
            flushLineBuffer();
        }
    }

    private synchronized void appendToLineBuffer(char c) {
        lineBuffer.append(c);
    }

    private synchronized void flushLineBuffer() {
        if (lineBuffer.length() > 0) {
            flushLine(lineBuffer.toString());
            lineBuffer.setLength(0);
        }
        flushAfter = nextFlushTime();
    }

    private void flushLine(String buffer) {
        if (stdout) {
            context.logOutput(buffer);
        } else {
            context.logError(buffer);
        }
    }

    @Override
    public synchronized void close() {
        flushTimerTask.cancel();
    }

    public static DefaultExecutionOutputHandler handleStdout(ExecutionContext context) {
        return new DefaultExecutionOutputHandler(context, true);
    }

    public static DefaultExecutionOutputHandler handleStderr(ExecutionContext context) {
        return new DefaultExecutionOutputHandler(context, false);
    }

}
