package com.xebialabs.deployit.plugin.remoting.vars;

import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import nl.javadude.scannit.Scannit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Multimap;
import com.google.common.collect.TreeMultimap;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereFile;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newIdentityHashMap;
import static com.google.common.collect.Sets.newTreeSet;

public abstract class VarsConverter {

    private static final String GENERATED_VARIABLE_PREFIX = "_pv";

    private static final String PROPERTIES_VARIABLE_NAME = "_properties";

    private static final char EMBEDDED_OBJECT_SEPARATOR = '_';

    private OverthereConnection connection;

    private Map<String, Object> vars;

    private Class<? extends Annotation> annotationClass;

    private Map<Object, String> alreadyConvertedCis = newIdentityHashMap();

    private int nextVarNumber = 1;

    private List<String> lines = newArrayList();

    private boolean uploadArtifactData = true;

    protected VarsConverter(OverthereConnection connection, Map<String, Object> vars, Class<? extends Annotation> annotationClass) {
        this.connection = connection;
        this.vars = vars;
        this.annotationClass = annotationClass;
    }

    protected VarsConverter(OverthereConnection connection, Map<String, Object> vars) {
        this(connection, vars, DerivedProperty.class);
    }

    public List<String> convert() {
        logger.debug("Starting vars conversion");
        TreeSet<String> sortedKeys = newTreeSet(vars.keySet());
        for (String varName : sortedKeys) {
            Object varValue = vars.get(varName);
            logger.debug("Converting variable [{}]", varName);
            convertVariable(varName, varValue);
        }
        logger.debug("Finished vars conversion");
        return lines;
    }

    public void setUploadArtifactData(boolean uploadArtifactData) {
        this.uploadArtifactData = uploadArtifactData;
    }

    private void convertVariable(String variableName, Object variableValue) {
        if (variableValue == null) {
            setNullVariable(variableName);
        } else if (variableValue instanceof Boolean) {
            setBooleanVariable(variableName, (Boolean) variableValue);
        } else if (variableValue instanceof Integer) {
            setIntegerVariable(variableName, (Integer) variableValue);
        } else if (variableValue instanceof Long) {
            setLongVariable(variableName, (Long) variableValue);
        } else if (variableValue instanceof ConfigurationItem) {
            alreadyConvertedCis.put(variableValue, variableName);
            appendConfigurationItemVar(variableName, (ConfigurationItem) variableValue);
        } else if (variableValue instanceof Collection) {
            setCollectionOfStringsVariable(variableName, (Collection<?>) variableValue);
        } else {
            setStringVariable(variableName, variableValue.toString());
        }
    }

    private void appendConfigurationItemVar(String varName, ConfigurationItem item) {
        Descriptor d = DescriptorRegistry.getDescriptor(item.getType());
        startCreateObject(varName);
        appendEmbeddedObjectDefinitions(varName, d);
        appendConfigurationItemVarMetaData(varName, item, d);
        appendRegularPropertiesToConfigurationItemVar(varName, item, d);
        appendDerivedPropertiesToConfigurationItemVar(varName, item);
        appendArtifactPropertiesToConfigurationItemVar(varName, item);
        endCreateObject(varName);
    }

    private void appendEmbeddedObjectDefinitions(String varName, Descriptor descriptor) {
        Multimap<String, String> embeddedObjects = getEmbeddedObjects(varName, descriptor);
        createEmbeddedObjects(embeddedObjects);
    }

    private static Multimap<String, String> getEmbeddedObjects(String varName, Descriptor descriptor) {
        Multimap<String, String> embeddedObjects = TreeMultimap.create();
        for (PropertyDescriptor pd : descriptor.getPropertyDescriptors()) {
            if (isEmbeddedObjectProperty(pd)) {
                embeddedObjects.put(varName + "." + getEmbeddedObjectName(pd), getEmbeddedPropertyName(pd));
            }
        }

        // Add embedded objects as properties of their parent object.
        for (String embeddedObjectName : embeddedObjects.keySet()) {
            String embeddedObjectParentName = embeddedObjectName.substring(0, embeddedObjectName.lastIndexOf("."));
            if (embeddedObjects.containsKey(embeddedObjectParentName)) {
                String nameOfEmbeddedObjectWithinParent = embeddedObjectName.substring(embeddedObjectName.lastIndexOf(".") + 1);
                embeddedObjects.get(embeddedObjectParentName).add(nameOfEmbeddedObjectWithinParent);
            }
        }

        return embeddedObjects;
    }

    private void createEmbeddedObjects(Multimap<String, String> objectAttributesKeyedByObjectName) {
        for (String embeddedObjectName : objectAttributesKeyedByObjectName.keySet()) {
            int lastDot = embeddedObjectName.lastIndexOf('.');
            createObjectAndSetObjectProperty(embeddedObjectName.substring(0, lastDot), embeddedObjectName.substring(lastDot + 1));
            setCollectionOfStringsProperty(embeddedObjectName, PROPERTIES_VARIABLE_NAME, objectAttributesKeyedByObjectName.get(embeddedObjectName));
        }
    }

    private static boolean isEmbeddedObjectProperty(PropertyDescriptor pd) {
        return getLastIndexOfEmbeddedObjectSeparator(pd) != -1;
    }

    private static String getEmbeddedObjectName(PropertyDescriptor pd) {
        return pd.getName().substring(0, getLastIndexOfEmbeddedObjectSeparator(pd)).replace(EMBEDDED_OBJECT_SEPARATOR, '.');
    }

    private static String getEmbeddedPropertyName(PropertyDescriptor pd) {
        return pd.getName().substring(getLastIndexOfEmbeddedObjectSeparator(pd) + 1);
    }

    private static int getLastIndexOfEmbeddedObjectSeparator(PropertyDescriptor pd) {
        return pd.getName().lastIndexOf(EMBEDDED_OBJECT_SEPARATOR);
    }

    private void appendConfigurationItemVarMetaData(String varName, ConfigurationItem item, Descriptor d) {
        setStringProperty(varName, "id", item.getId());
        setStringProperty(varName, "name", item.getName());
        setStringProperty(varName, "type", item.getType().toString());

        SortedSet<String> properties = newTreeSet();
        properties.add("id");
        properties.add("name");
        properties.add("type");
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            String propertyName;
            if (!isEmbeddedObjectProperty(pd)) {
                propertyName = pd.getName();
            } else {
                propertyName = pd.getName().substring(0, pd.getName().indexOf(EMBEDDED_OBJECT_SEPARATOR));
            }
            properties.add(propertyName);
        }

        Set<Method> methodsAnnotatedWith = Scannit.getInstance().getMethodsAnnotatedWith(annotationClass);
        Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
        for (Method method : methodsAnnotatedWith) {
            if (descriptor.isAssignableTo(method.getDeclaringClass())) {
                properties.add(getDerivedPropertyName(method));
            }
        }

        setCollectionOfStringsProperty(varName, PROPERTIES_VARIABLE_NAME, properties);
    }

    protected String getDerivedPropertyName(Method method) {
        try {
            Annotation derivedProperty = method.getAnnotation(annotationClass);
            Method valueMethod = annotationClass.getMethod("value");
            return (String) valueMethod.invoke(derivedProperty);
        } catch (Exception exc) {
            throw new RuntimeException("Cannot get name of @DerivedProperty " + method, exc);
        }
    }

    @SuppressWarnings("unchecked")
    private void appendRegularPropertiesToConfigurationItemVar(String varName, ConfigurationItem item, Descriptor d) {
        for (PropertyDescriptor pd : d.getPropertyDescriptors()) {
            String objectName;
            String propertyName;
            if (!isEmbeddedObjectProperty(pd)) {
                objectName = varName;
                propertyName = pd.getName();
            } else {
                objectName = varName + "." + getEmbeddedObjectName(pd);
                propertyName = getEmbeddedPropertyName(pd);
            }
            Object propertyValue = pd.get(item);

            switch (pd.getKind()) {
            case BOOLEAN:
                if (propertyValue == null) {
                    setNullProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof Boolean)) {
                    throw new IllegalStateException("Property " + pd + " is not a Boolean but a " + propertyValue.getClass().getName());
                } else {
                    setBooleanProperty(objectName, propertyName, (Boolean) propertyValue);
                }
                break;
            case INTEGER:
                if (propertyValue == null) {
                    setNullProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof Integer)) {
                    throw new IllegalStateException("Property " + pd + " is not an Integer but a " + propertyValue.getClass().getName());
                } else {
                    setIntegerProperty(objectName, propertyName, (Integer) propertyValue);
                }
                break;
            case STRING:
            case ENUM:
                if (propertyValue == null) {
                    setNullProperty(objectName, propertyName);
                } else if (pd.isPassword()) {
                    setPasswordProperty(objectName, propertyName, propertyValue.toString());
                } else {
                    setStringProperty(objectName, propertyName, propertyValue.toString());
                }
                break;
            case LIST_OF_STRING:
                if (propertyValue == null) {
                    setEmptyCollectionProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof List)) {
                    throw new IllegalStateException("Property " + pd + " is not a List but a " + propertyValue.getClass().getName());
                } else {
                    setCollectionOfStringsProperty(objectName, propertyName, (List<String>) propertyValue);
                }
                break;
            case SET_OF_STRING:
                if (propertyValue == null) {
                    setEmptyCollectionProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof Set)) {
                    throw new IllegalStateException("Property " + pd + " is not a Set but a " + propertyValue.getClass().getName());
                } else {
                    setCollectionOfStringsProperty(objectName, propertyName, (Set<String>) propertyValue);
                }
                break;
            case CI:
                if (d.isAssignableTo(Type.valueOf(Deployed.class)) && pd.getName().equals(Deployed.DEPLOYABLE_FIELD)) {
                    setNullProperty(objectName, propertyName);
                } else if (propertyValue == null) {
                    setNullProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof ConfigurationItem)) {
                    throw new IllegalStateException("Property " + pd + " is not a ConfigurationItem but a " + propertyValue.getClass().getName());
                } else {
                    setCiReferenceProperty(objectName, propertyName, (ConfigurationItem) propertyValue);
                }
                break;
            case LIST_OF_CI:
                if (propertyValue == null) {
                    setEmptyCollectionProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof List)) {
                    throw new IllegalStateException("Property " + pd + " is not a List but a " + propertyValue.getClass().getName());
                } else {
                    setCollectionOfCiReferencesProperty(objectName, propertyName, (List<ConfigurationItem>) propertyValue);
                }
                break;
            case SET_OF_CI:
                if (propertyValue == null) {
                    setEmptyCollectionProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof Set)) {
                    throw new IllegalStateException("Property " + pd + " is not a Set but a " + propertyValue.getClass().getName());
                } else {
                    setCollectionOfCiReferencesProperty(objectName, propertyName, (Set<ConfigurationItem>) propertyValue);
                }
                break;
            case MAP_STRING_STRING:
                if (propertyValue == null) {
                    setEmptyMapProperty(objectName, propertyName);
                } else if (!(propertyValue instanceof Map)) {
                    throw new IllegalStateException("Property " + pd + " is not a Map but a " + propertyValue.getClass().getName());
                } else {
                    setMapOfStringToStringReferencesProperty(objectName, propertyName, (Map<String, String>) propertyValue);
                }
                break;
            default:
                throw new IllegalStateException("Should not end up here!");
            }
        }
    }

    private void appendDerivedPropertiesToConfigurationItemVar(String varName, ConfigurationItem item) {
        Set<Method> methodsAnnotatedWith = Scannit.getInstance().getMethodsAnnotatedWith(annotationClass);
        Descriptor descriptor = DescriptorRegistry.getDescriptor(item.getType());
        for (Method method : methodsAnnotatedWith) {
            if (descriptor.isAssignableTo(method.getDeclaringClass())) {
                try {
                    convertProperty(varName, getDerivedPropertyName(method), method.invoke(item));
                } catch (IllegalAccessException e) {
                    throw new RuntimeException("Method " + method.getName() + " in " + method.getDeclaringClass() + " cannot be accessed.", e);
                } catch (InvocationTargetException e) {
                    throw new RuntimeException("Method " + method.getName() + " in " + method.getDeclaringClass() + " cannot be invoked.", e);
                }
            }
        }
    }

    private void convertProperty(String objectName, String propertyName, Object propertyValue) {
        if (propertyValue == null) {
            setNullProperty(objectName, propertyName);
        } else if (propertyValue instanceof Boolean) {
            setBooleanProperty(objectName, propertyName, (Boolean) propertyValue);
        } else if (propertyValue instanceof Integer) {
            setIntegerProperty(objectName, propertyName, (Integer) propertyValue);
        } else if (propertyValue instanceof Long) {
            setIntegerProperty(objectName, propertyName, (int) (long) (Long) propertyValue);
        } else if (propertyValue instanceof ConfigurationItem) {
            setCiReferenceProperty(objectName, propertyName, (ConfigurationItem) propertyValue);
        } else if (propertyValue instanceof Collection) {
            setCollectionOfStringsProperty(objectName, propertyName, (Collection<?>) propertyValue);
        } else {
            setStringProperty(objectName, propertyName, propertyValue.toString());
        }
    }

    private void appendArtifactPropertiesToConfigurationItemVar(String varName, ConfigurationItem item) {
        if (item instanceof Artifact) {
            if (!uploadArtifactData) {
                logger.debug("Setting file property of [{}] to null because automatic uploading of artifact data is explicitly turned off", varName);
                setNullProperty(varName, "file");
                return;
            }

            if(!vars.containsKey(varName)) {
                logger.debug("Setting file property of [{}] to null because it is an indirectly referenced object", varName);
                setNullProperty(varName, "file");
                return;
            }

            Artifact a = (Artifact) item;
            if (a.getFile() == null) {
                logger.debug("Setting file property of [{}] to null because artifact has a null file reference.", varName);
                setNullProperty(varName, "file");
                return;
            }

            logger.debug("Creating temporary file for variable [{}]", varName);
            OverthereFile uploadedFileArtifact = connection.getTempFile(a.getFile().getName());
            logger.debug("Uploading artifact for variable [{}] to [{}]", varName, uploadedFileArtifact.getPath());
            a.getFile().copyTo(uploadedFileArtifact);
            logger.debug("Uploaded artifact for variable [{}] to [{}]", varName, uploadedFileArtifact.getPath());
            setStringProperty(varName, "file", uploadedFileArtifact.getPath());
        }
    }

    protected String getConfigurationItemVariableName(ConfigurationItem item) {
        if (alreadyConvertedCis.containsKey(item)) {
            return alreadyConvertedCis.get(item);
        }

        String pvName = generateUniqueVariableName();
        alreadyConvertedCis.put(item, pvName);

        appendConfigurationItemVar(pvName, item);
        return pvName;
    }

    protected String generateUniqueVariableName() {
        return GENERATED_VARIABLE_PREFIX + nextVarNumber++;
    }

    protected void add(String line) {
        lines.add(line);
    }

    protected abstract void setNullVariable(String variableValue);

    protected abstract void setBooleanVariable(String variableValue, boolean propertyValue);

    protected abstract void setIntegerVariable(String variableValue, int propertyValue);

    protected abstract void setLongVariable(String variableValue, long propertyValue);

    protected abstract void setStringVariable(String variableValue, String propertyValue);

    protected abstract void setCollectionOfStringsVariable(String variableValue, Collection<?> propertyValue);

    protected abstract void startCreateObject(String objectName);

    protected abstract void endCreateObject(String objectName);

    protected abstract void setNullProperty(String objectName, String propertyName);

    protected abstract void setEmptyCollectionProperty(String objectName, String propertyName);

    protected abstract void setEmptyMapProperty(String objectName, String propertyName);

    protected abstract void setBooleanProperty(String objectName, String propertyName, boolean propertyValue);

    protected abstract void setIntegerProperty(String objectName, String propertyName, int propertyValue);

    protected abstract void setStringProperty(String objectName, String propertyName, String propertyValue);

    protected abstract void setPasswordProperty(String objectName, String propertyName, String propertyValue);

    protected abstract void setCollectionOfStringsProperty(String objectName, String propertyName, Collection<?> propertyValue);

    protected abstract void setCiReferenceProperty(String objectName, String propertyName, ConfigurationItem propertyValue);

    protected abstract void setCollectionOfCiReferencesProperty(String objectName, String propertyName, Collection<ConfigurationItem> propertyValue);

    protected abstract void setMapOfStringToStringReferencesProperty(String objectName, String propertyName, Map<String, String> propertyValue);

    protected abstract void createObjectAndSetObjectProperty(String objectName, String propertyName);

    private static final Logger logger = LoggerFactory.getLogger(VarsConverter.class);

}
