package com.xebialabs.deployit.plugin.overthere.step;

import com.xebialabs.deployit.plugin.api.flow.ExecutionContext;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.flow.StepExitCode;
import com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereConnection;
import com.xebialabs.overthere.OverthereExecutionOutputHandler;

import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStderr;
import static com.xebialabs.deployit.plugin.overthere.DefaultExecutionOutputHandler.handleStdout;
import static com.xebialabs.overthere.ConnectionOptions.ADDRESS;
import static com.xebialabs.overthere.OperatingSystemFamily.WINDOWS;

@SuppressWarnings("serial")
public class CheckCommandExecutionStep implements Step {

    private Host host;

    private String hostname;

    public CheckCommandExecutionStep(Host host) {
        this.host = host;
        if (host.hasProperty(ADDRESS)) {
            this.hostname = host.getProperty(ADDRESS);
        } else {
            this.hostname = host.getName();
        }
    }

    @Override
    public String getDescription() {
        return "Check whether XL Deploy can execute commands on " + hostname;
    }

    @Override
    public int getOrder() {
        return 50;
    }

    @Override
    public StepExitCode execute(ExecutionContext ctx) throws Exception {
        try (DefaultExecutionOutputHandler stdoutHandler = handleStdout(ctx);
             DefaultExecutionOutputHandler stderrHandler = handleStderr(ctx);
             OverthereConnection connection = host.getConnection()) {
            ctx.logOutput("Listing the contents of the temporary directory...");
            int i = echoTmpDirContents(connection, stdoutHandler, stderrHandler);
            if (i != 0) {
                ctx.logError("Failed to execute command on " + hostname + ". Return code was [" + i + "]. Please check the logs.");
                return StepExitCode.FAIL;
            }
            ctx.logOutput("Succesfully executed commands on " + hostname + ".");
            return StepExitCode.SUCCESS;
        } catch (Exception exc) {
            ctx.logError("Cannot execute commands on " + hostname + ". Please check the connection settings.", exc);
            return StepExitCode.FAIL;
        }
    }

    private int echoTmpDirContents(OverthereConnection connection, OverthereExecutionOutputHandler stdoutHandler, OverthereExecutionOutputHandler stderrHandler) {
        CmdLine cmdLine = new CmdLine();
        if (connection.getHostOperatingSystem() == WINDOWS) {
            cmdLine.addArgument("cmd");
            cmdLine.addArgument("/c");
            cmdLine.addArgument("dir");
        } else {
            cmdLine.addArgument("ls");
        }

        String tempDir = host.getTemporaryDirectoryPath();
        if (tempDir == null) {
            tempDir = connection.getHostOperatingSystem().getDefaultTemporaryDirectoryPath();
        }

        cmdLine.addArgument(tempDir);
        return connection.execute(stdoutHandler, stderrHandler, cmdLine);
    }
}
