package com.xebialabs.deployit.plugin.tomcat.freemarker;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import com.xebialabs.deployit.plugin.generic.step.BaseDeploymentStep;

import freemarker.ext.beans.BeanModel;
import freemarker.ext.beans.BeansWrapper;
import freemarker.ext.dom.NodeModel;
import freemarker.template.TemplateModel;
import freemarker.template.TemplateModelException;

import static com.google.common.base.Strings.emptyToNull;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

public class ContextXmlHelper {
    private BaseDeploymentStep step;

    public ContextXmlHelper(BaseDeploymentStep step) {
            this.step = step;
        }

    public NodeModel readOrCreateContext(String path) {
        if (!step.hostFileExists(path)) {
            step.createOrReplaceHostFile("<Context></Context>",path);
        }
        return step.readHostXmlFileAsModel(path);
    }

    public String attributeMapToString(Map<String,String> map) {
        StringBuilder builder = new StringBuilder();
        for(Map.Entry<String,String> entry : map.entrySet()) {
            builder.append(entry.getKey()).append("=\"").append(entry.getValue()).append("\" ");
        }
        return builder.toString();
    }

    public Map<String,String> resolveElementAttributes(NodeModel nodeModel, BeanModel ci, Map<String,String> propertyToAttributeMapping, boolean isDestroyOperation) throws TemplateModelException {
        Map<String,String> attributes = newHashMap();
        if (isDestroyOperation) {
            List<String> attrNames = resolveAttributeNamesFromPropertyToAttributeMapping(propertyToAttributeMapping);
            attributes.putAll(removeAttributes(nodeModel,attrNames));
        } else {
            Map<String, String> elementAttrsFromCi = populateElementAttributesFromCi(ci, propertyToAttributeMapping);
            attributes.putAll(mergeWithElementAttributes(nodeModel, elementAttrsFromCi));
        }

        return attributes;
    }

    @SuppressWarnings("unchecked")
    public Map<String,String> populateElementAttributesFromCi(final BeanModel ci, Map<String,String> propertyToAttributeMapping) throws TemplateModelException {
        Map<String,String> attributes = newHashMap();
        BeansWrapper beansWrapper = BeansWrapper.getDefaultInstance();
        for (Map.Entry<String, String> entry : propertyToAttributeMapping.entrySet()) {
            String propertyName = entry.getKey();
            String attributeName = emptyToNull(entry.getValue()) == null ? propertyName : entry.getValue();
            TemplateModel tm = ci.get(propertyName);
            Object value = beansWrapper.unwrap(tm);
            if (value instanceof Map) {
                //actually only used for dbcp connectionProperties.
                value = convertMapToTomcatMapStringFormat((Map<String, String>) value);
            }
            if (value != null) {
                attributes.put(attributeName, value.toString());
            }
        }
        return attributes;
    }

    protected List<String> resolveAttributeNamesFromPropertyToAttributeMapping(Map<String,String> propertyToAttributeMapping) {
        List<String> attributes = newArrayList();
        for (Map.Entry<String, String> entry : propertyToAttributeMapping.entrySet()) {
            String propertyName = entry.getKey();
            String attributeName = emptyToNull(entry.getValue()) == null ? propertyName : entry.getValue();
            attributes.add(attributeName);
        }
        return attributes;
    }

    protected String convertMapToTomcatMapStringFormat(Map<String, String> map) {
        StringBuilder builder = new StringBuilder();
        for (Map.Entry<String, String> entry : map.entrySet()) {
            builder.append(entry.getKey()).append('=').append(entry.getValue()).append(';');
        }
        return builder.toString();
    }

    protected Map<String,String> mergeWithElementAttributes(NodeModel nodeModel, Map<String,String> overrideAttributes) throws TemplateModelException {
        Node node = nodeModel.getNode();
        Map<String,String> mergedAttributes = newHashMap();
        NamedNodeMap attributes = node.getAttributes();
        int size = attributes == null ? 0 : attributes.getLength();
        for(int i=0; i<size;i++) {
            Node attr = attributes.item(i);
            mergedAttributes.put(attr.getNodeName(), attr.getNodeValue());
        }
        mergedAttributes.putAll(overrideAttributes);
        return mergedAttributes;
    }

    protected Map<String,String> removeAttributes(NodeModel nodeModel, Collection<String> attributesToRemove) throws TemplateModelException {
        Node node = nodeModel.getNode();
        Map<String,String> mergedAttributes = newHashMap();
        NamedNodeMap attributes = node.getAttributes();
        int size = attributes == null ? 0 : attributes.getLength();
        for(int i=0; i<size;i++) {
            Node attr = attributes.item(i);
            if(attributesToRemove.contains(attr.getNodeName())) {
                continue;
            }
            mergedAttributes.put(attr.getNodeName(), attr.getNodeValue());
        }
        return mergedAttributes;
    }
}
