package com.xebialabs.deployit.plugin.was.deployed;

import java.util.List;
import java.util.Map;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;

import com.xebialabs.deployit.plugin.api.deployment.planning.Create;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.Destroy;
import com.xebialabs.deployit.plugin.api.deployment.planning.Modify;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionContext;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.python.PythonManagedDeployed;
import com.xebialabs.deployit.plugin.was.container.WasContainer;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.DESTROY;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedHelper.syncNodes;

@SuppressWarnings("serial")
@Metadata(description = "Base class for all extensible deployed configuration items.")
public abstract class ExtensibleDeployed<D extends Deployable> extends PythonManagedDeployed<D, WasContainer> {

    @Property(hidden = true, required = false, description = "Python script invoked to deploy a Java EE artifact or create a Java EE resource")
    private String createScript;

    @Property(hidden = true, defaultValue = "Create", description = "The word is used to prefix a step description for the create operation.")
    private String createVerb;

    @Property(hidden = true, defaultValue = "60", description = "The order in which a create step will be executed.")
    private int createOrder;

    @Property(hidden = true, defaultValue = "65", description = "The order in which a synchronize after create step will be executed.")
    private int syncAfterCreateOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to upgrade a Java EE artifact or modify a Java EE resource")
    private String modifyScript;

    @Property(hidden = true, defaultValue = "Modify", description = "The word is used to prefix a step description for the modify operation.")
    private String modifyVerb;

    @Property(hidden = true, defaultValue = "60", description = "The order in which a modify step will be executed.")
    private int modifyOrder;

    @Property(hidden = true, defaultValue = "65", description = "The order in which a synchronize after modify step will be executed.")
    private int syncAfterModifyOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to undeploy a Java EE artifact or destroy a Java EE resource")
    private String destroyScript;

    @Property(hidden = true, defaultValue = "Destroy", description = "The word is used to prefix a step description for the destroy operation.")
    private String destroyVerb;

    @Property(hidden = true, defaultValue = "40", description = "The order in which a destroy step will be executed.")
    private int destroyOrder;

    @Property(hidden = true, defaultValue = "45", description = "The order in which a synchronize after destroy step will be executed.")
    private int syncAfterDestroyOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to start a Java EE artifact or Java EE resource")
    private String startScript;

    @Property(hidden = true, defaultValue = "Start", description = "The word is used to prefix a step description for the start operation.")
    private String startVerb;

    @Property(hidden = true, defaultValue = "50", description = "The order in which a start step will be executed.")
    private int startOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to stop a Java EE artifact or Java EE resource")
    private String stopScript;

    @Property(hidden = true, defaultValue = "Stop", description = "The word is used to prefix a step description for the stop operation.")
    private String stopVerb;

    @Property(hidden = true, defaultValue = "50", description = "The order in which a synchronize after modify stop will be executed.")
    private int stopOrder;

    @Property(hidden = true, required = false, description = "Python script invoked to inspect a Java EE artifact or Java EE resource")
    private String inspectScript;

    @Property(hidden = true, defaultValue = "Inspect", description = "The word is used to prefix a step description for the inspect operation.")
    private String inspectVerb;

    protected final Map<String, Object> context = ImmutableMap.of("deployed", (Object) this);

    @SuppressWarnings("unused")
    @Property(hidden = true, defaultValue = "id, type, properties, deployable, container, createOrder, createScript, createVerb, " +
            "syncAfterCreateOrder, modifyOrder, modifyScript, modifyVerb, syncAfterModifyOrder, destroyOrder, destroyScript, " +
            "securityPermissions, inheritPermissions, exposeDeployedApplication, destroyVerb, syncAfterDestroyOrder, startOrder, startScript, startVerb, stopOrder, " +
            "stopScript, stopVerb, inspectScript, inspectVerb, discoverScript, discoverOrder, libraryScripts",
            description = "Standard properties that are not exposed to any python wsadmin script.")
    private String standardPropertiesNotToExpose;

    @Inspect
    public void inspect(InspectionContext ctx) {
        addStep(ctx, inspectScript, inspectVerb);
    }

    @Create
    public void create(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, createOrder, createScript, createVerb, checkpoint(delta, CREATE));
        syncNodes(ctx, syncAfterCreateOrder, getContainer());
        addStep(ctx, startOrder, startScript, startVerb, false);
    }

    @SuppressWarnings("rawtypes")
    @Modify
    public void modify(DeploymentPlanningContext ctx, Delta delta) {
        if (addStep(ctx, modifyOrder, modifyScript, modifyVerb, checkpoint(delta, MODIFY))) {
            syncNodes(ctx, syncAfterModifyOrder, getContainer());
        } else {
            ExtensibleDeployed previous = (ExtensibleDeployed) delta.getPrevious();
            ExtensibleDeployed current = (ExtensibleDeployed) delta.getDeployed();
            previous.destroy(ctx, delta);
            current.create(ctx, delta);
        }
    }

    @Destroy
    public void destroy(DeploymentPlanningContext ctx, Delta delta) {
        addStep(ctx, stopOrder, stopScript, stopVerb, false);
        addStep(ctx, destroyOrder, destroyScript, destroyVerb, false, checkpoint(delta, DESTROY));
        syncNodes(ctx, syncAfterDestroyOrder, getContainer());
    }

    public int getSyncAfterCreateOrder() {
        return syncAfterCreateOrder;
    }

    public List<Step> controlTaskDispatch(String name) {
        PropertyDescriptor scriptProperty = DescriptorRegistry.getDescriptor(getType()).getPropertyDescriptor(name + "PythonScript");
        checkArgument(scriptProperty != null, "Control task script property '%sPythonScript' not defined for CI type %s", name, getType());
        return ImmutableList.<Step>of(new PythonDeploymentStep(1, this.getContainer().getCell(),
                ((String) scriptProperty.get(this)) + ".py", context, "Executing " + name));
    }
}
