/*
 * @(#)SqlScriptOrderer.java     1 Sep 2011
 *
 * Copyright © 2010 Andrew Phillips.
 *
 * ====================================================================
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or 
 * implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * ====================================================================
 */
package com.xebialabs.deployit.community.wasext.planning;


import static com.google.common.base.Predicates.or;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.filter;
import static com.xebialabs.deployit.community.wasext.util.Predicates.instanceOf;

import java.util.Set;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSet.Builder;
import com.xebialabs.deployit.plugin.api.deployment.planning.SingleTypeContributor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.was.container.ManagedWebServer;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedArtifact;

public abstract class ReferencedWebserverContributor extends SingleTypeContributor<ExtensibleDeployedArtifact> {
    private static final String WEBSERVERS_PROPERTY = "webServers";
    private static final Predicate<ConfigurationItem> IS_EAR_OR_WAR_MODULE =
        or(instanceOf(Type.valueOf("was.EarModule")), 
           instanceOf(Type.valueOf("was.WarModule")));
    
    public ReferencedWebserverContributor() {
        super(ExtensibleDeployedArtifact.class);
    }
    
    protected Set<ManagedWebServer> getReferencedWebservers() {
        Builder<ManagedWebServer> webservers = ImmutableSet.builder();
        
        // gather all referenced webservers
        for (ExtensibleDeployedArtifact earOrWarModule : getEarAndWarModules()) {
            Set<ManagedWebServer> referencedWebservers = earOrWarModule.getProperty(WEBSERVERS_PROPERTY);
            if (referencedWebservers != null) {
                webservers.addAll(referencedWebservers);
            }
        }
        return webservers.build();
    }

    private Set<ExtensibleDeployedArtifact> getEarAndWarModules() {
        Builder<ExtensibleDeployedArtifact> earAndWarModules = ImmutableSet.builder();
        earAndWarModules.addAll(filter(concat(deployedsCreated, deployedsRemoved), IS_EAR_OR_WAR_MODULE));
        Iterable<TypedDelta> earAndWarModuleDeltas = filter(concat(deployedsModified, deployedsNoop), 
                new Predicate<TypedDelta>() {
                    @Override
                    public boolean apply(TypedDelta input) {
                        return IS_EAR_OR_WAR_MODULE.apply(input.getDeployed());
                    }
                });
        for (TypedDelta earOrWarModuleDelta : earAndWarModuleDeltas) {
            earAndWarModules.add(earOrWarModuleDelta.getPrevious());
            earAndWarModules.add(earOrWarModuleDelta.getDeployed());
        }
        return earAndWarModules.build();
    }
}