package com.xebialabs.deployit.community.wasext.planning;


import static com.google.common.base.Functions.compose;
import static com.google.common.collect.Iterables.concat;
import static com.google.common.collect.Iterables.transform;
import static com.xebialabs.deployit.community.wasext.planning.StopAndStartColdDeploymentContainers.isColdDeployContainer;
import static com.xebialabs.deployit.community.wasext.util.Predicates.cast;
import static com.xebialabs.deployit.community.wasext.util.Predicates.extractDeployed;
import static com.xebialabs.deployit.community.wasext.util.Predicates.instanceOf;
import static java.lang.Boolean.TRUE;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSet.Builder;
import com.xebialabs.deployit.community.wasext.task.ClusterTasks;
import com.xebialabs.deployit.community.wasext.task.ManagedServerTasks;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.planning.SingleTypeContributor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.was.container.Cluster;
import com.xebialabs.deployit.plugin.was.container.ManagedServer;
import com.xebialabs.deployit.plugin.was.container.WasContainer;
import com.xebialabs.deployit.plugin.was.deployed.ExtensibleDeployedResource;

public class RestartReferencedContainers extends SingleTypeContributor<ExtensibleDeployedResource> {
    private static final String CONTAINER_RESTART_REQUIRED_PROPERTY = "containerRestartRequired";
    private static final Predicate<ConfigurationItem> IS_CLUSTER = new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                return instanceOf(Type.valueOf(Cluster.class)).apply(input);
            }
        };
    private static final Predicate<ConfigurationItem> IS_MANAGED_SERVER = new Predicate<ConfigurationItem>() {
            @Override
            public boolean apply(ConfigurationItem input) {
                return instanceOf(Type.valueOf(ManagedServer.class)).apply(input);
            }
        };

    private static final Logger LOGGER = LoggerFactory.getLogger(RestartReferencedContainers.class);

    public RestartReferencedContainers() {
        super(ExtensibleDeployedResource.class);
    }

    @Contributor
    public void restartContainers(Deltas deltas, DeploymentPlanningContext ctx) {
        filterDeltas(deltas.getDeltas());

        for (WasContainer container : getReferencedContainers()) {
            // a "cold deploy" container already triggers restart steps
            if (!isColdDeployContainer(container)) {
                restart(container, ctx);
            }
        }
    }

    private Set<WasContainer> getReferencedContainers() {
        Builder<WasContainer> containers = ImmutableSet.builder();
        Iterable<ExtensibleDeployedResource> changedDeployeds = concat(deployedsCreated, 
                transform(deployedsModified, compose(cast(ExtensibleDeployedResource.class), extractDeployed())),
                deployedsRemoved);
        for (ExtensibleDeployedResource deployed : changedDeployeds) {
            // may be null
            if (deployed.hasProperty(CONTAINER_RESTART_REQUIRED_PROPERTY) 
                    && TRUE.equals(deployed.getProperty(CONTAINER_RESTART_REQUIRED_PROPERTY))) {
                containers.add(deployed.getContainer());
            }
        }
        return containers.build();
    }

    protected void restart(WasContainer container, DeploymentPlanningContext ctx) {
        if (IS_CLUSTER.apply(container)) {
            ClusterTasks.restart((Cluster) container, ctx);
        } else if (IS_MANAGED_SERVER.apply(container)) {
            ManagedServerTasks.restart((ManagedServer) container, ctx);
        } else {
            LOGGER.warn("Unable to generate automatic restart steps for unsupported WebSphere container type '{}'", 
                    Type.valueOf(container.getClass()));
        }
    }
}
