/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.mapper;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.ci.artifact.ConfigurationFiles;
import com.xebialabs.deployit.ci.artifact.Folder;
import com.xebialabs.deployit.ci.artifact.Libraries;
import com.xebialabs.deployit.ci.artifact.mapping.ConfigurationFilesMapping;
import com.xebialabs.deployit.ci.artifact.mapping.LibrariesMapping;
import com.xebialabs.deployit.mapper.ModificationSupportingStepGeneratingMapper;
import com.xebialabs.deployit.plugin.was.ci.WasScope;
import com.xebialabs.deployit.plugin.was.ci.WasSharedLibrary;
import com.xebialabs.deployit.plugin.was.ci.WasSharedLibraryToWasScopeTargetMapping;
import com.xebialabs.deployit.plugin.was.step.CreateWasSharedLibraryStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasSharedLibraryStep;
import com.xebialabs.deployit.plugin.was.step.UpdateWasSharedLibraryStep;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.util.List;
import java.util.Set;

public class SharedLibraryToWasScopeTargetMapper extends
        ModificationSupportingStepGeneratingMapper<WasSharedLibrary, WasSharedLibraryToWasScopeTargetMapping, WasScope> {

	public SharedLibraryToWasScopeTargetMapper(Change<Deployment> change) {
		super(change, false);
	}

	@Override
	protected void generateAdditionStepsForAddedMapping(WasSharedLibrary sharedLibrary, WasSharedLibraryToWasScopeTargetMapping newMapping,
	        WasScope scopeTarget, List<Step> steps) {

		String remoteHostFileSeparator = scopeTarget.getCell().getCellHost().getFileSeparator();
		Deployment deployment = change.getNewRevision();
		String completeClassPath = prepareCompleteClassPath(sharedLibrary, deployment, remoteHostFileSeparator);
		steps.add(new CreateWasSharedLibraryStep(sharedLibrary, scopeTarget, completeClassPath));
	}

	@Override
	protected void generateDeletionStepsForDeletedMapping(WasSharedLibrary sharedLibrary, WasSharedLibraryToWasScopeTargetMapping oldVersionOfModifiedMapping,
	        WasScope scopeTarget, List<Step> steps) {
		steps.add(new DestroyWasSharedLibraryStep(sharedLibrary, scopeTarget));
	}

	@Override
	protected void generateModificationStepsForModifiedMapping(final WasSharedLibrary oldSharedLibrary,
	        final WasSharedLibraryToWasScopeTargetMapping oldMapping, final WasScope oldMappingTarget, final WasSharedLibrary newSharedLibrary,
	        final WasSharedLibraryToWasScopeTargetMapping newMapping, final WasScope newMappingTarget, final List<Step> steps) {

		String oldClasspath = prepareCompleteClassPath(oldSharedLibrary, change.getOldRevision(), oldMappingTarget.getCell().getCellHost().getFileSeparator());
		String newClasspath = prepareCompleteClassPath(newSharedLibrary, change.getNewRevision(), newMappingTarget.getCell().getCellHost().getFileSeparator());

		if (!oldClasspath.equals(newClasspath)) {
			steps.add(new UpdateWasSharedLibraryStep(newSharedLibrary, newMappingTarget, newClasspath));
		}

	}

	private String prepareCompleteClassPath(WasSharedLibrary sharedLibrary, Deployment deployment, String pathSeparator) {
		List<String> strings = Lists.newArrayList();
		CollectionUtils.addIgnoreNull(strings, sharedLibrary.getClassPath());
		if (CollectionUtils.isNotEmpty(sharedLibrary.getConfigurationFiles())) {
			CollectionUtils.addIgnoreNull(strings, getClassPathEntriesForConfigFiles(deployment, sharedLibrary.getConfigurationFiles(), pathSeparator));
		}

		if (CollectionUtils.isNotEmpty(sharedLibrary.getLibraries())) {
			CollectionUtils.addIgnoreNull(strings, getClassPathEntriesForLibraries(deployment, sharedLibrary.getLibraries(), pathSeparator));
		}

		return StringUtils.join(strings, ";");
	}

	private String getClassPathEntriesForConfigFiles(Deployment deployment, Set<ConfigurationFiles> configFiles, String remoteHostFileSeparator) {
		Set<String> classPathEntries = Sets.newHashSet();
		List<ConfigurationFilesMapping> mappingsForSource = deployment.getMappingsOfType(ConfigurationFilesMapping.class);
		for (ConfigurationFilesMapping eachMapping : mappingsForSource) {
			if (configFiles.contains(eachMapping.getSource())) {
				String targetDirectory = eachMapping.getTargetDirectory();
				classPathEntries.add(targetDirectory);
			}
		}
		return StringUtils.join(classPathEntries, ";");
	}

	private String getClassPathEntriesForLibraries(Deployment deployment, Set<Libraries> libraries, String remoteHostFileSeparator) {
		Set<String> classPathEntries = Sets.newHashSet();
		List<LibrariesMapping> mappingsForSource = deployment.getMappingsOfType(LibrariesMapping.class);
		for (LibrariesMapping eachMapping : mappingsForSource) {
			if (libraries.contains(eachMapping.getSource())) {
				String targetDirectory = eachMapping.getTargetDirectory();
				Folder library = eachMapping.getSource();
				if (CollectionUtils.isEmpty(library.getFiles())) {
					classPathEntries.add(targetDirectory + remoteHostFileSeparator + library.getName());
				} else {
					// it's a directory, add the filename recursively
					Set<File> folderFiles = Sets.newHashSet(library.getFiles());
					addFileNamesToClassPathEntries(classPathEntries, folderFiles, targetDirectory, remoteHostFileSeparator);
				}
			}
		}
		return StringUtils.join(classPathEntries, ";");
	}

	private void addFileNamesToClassPathEntries(Set<String> classPathEntries, Set<File> files, String targetDirectory, String pathSeparator) {
        for (File eachFile : files) {
            if (eachFile.isDirectory()) {
                String modifiedTargetDirectory = targetDirectory + pathSeparator + eachFile.getName();
                addFileNamesToClassPathEntries(classPathEntries, Sets.newHashSet(eachFile.listFiles()), modifiedTargetDirectory, pathSeparator);
            } else {
                classPathEntries.add(targetDirectory + pathSeparator + eachFile.getName());
            }
        }
	}
}
