/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.was.runbook;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;

import com.xebialabs.deployit.Change;
import com.xebialabs.deployit.ChangePlan;
import com.xebialabs.deployit.RunBook;
import com.xebialabs.deployit.Step;
import com.xebialabs.deployit.ci.Deployment;
import com.xebialabs.deployit.plugin.was.ci.WasCluster;
import com.xebialabs.deployit.plugin.was.ci.WasNodeAgent;
import com.xebialabs.deployit.plugin.was.ci.WasManagedServer;
import com.xebialabs.deployit.plugin.was.step.AddSingleWasClusterMemberStep;
import com.xebialabs.deployit.plugin.was.step.CreateWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.DestroyWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.ModifyWasServerStep;
import com.xebialabs.deployit.plugin.was.step.RemoveWasClusterMemberStep;
import com.xebialabs.deployit.plugin.was.step.StartWasManagedServerStep;
import com.xebialabs.deployit.plugin.was.step.StopWasClusterStep;
import com.xebialabs.deployit.plugin.was.step.SynchronizeWasNodeStep;
import com.xebialabs.deployit.reflect.ConfigurationItemReflectionUtils;
import com.xebialabs.deployit.util.SingleTypeHandlingRunBook;

/**
 * Supports the creation, modification and deletion of a {@link com.xebialabs.deployit.plugin.was.ci.WasCluster WAS Cluster}.
 * 
 * <h4>Conditions</h4>
 * 
 * Will trigger if the change plan contains the addition, modification or deletion of a {@link WasCluster WAS Cluster} CI.
 * 
 * <h4>Actions</h4>
 * 
 * Addition:
 * 
 * <ol>
 * <li>Create cluster in the cell
 * <li>For every added server, create it
 * <li>For every added server, synchronize node
 * <li>For every added server, start it
 * </ol>
 * 
 * Modification:
 * 
 * <ol>
 * <li>The cluster is stopped
 * <li>Remove all deleted servers
 * <li>Modify remaining cluster members
 * <li>For every added server, create it
 * <li>For every added server, synchronize node
 * <li>For every added server, start it
 * </ol>
 * 
 * Deletion:
 * 
 * <ol>
 * <li>Stop the cluster
 * <li>Destroy the cluster
 * </ol>
 */
public class WasClusterRunBook extends SingleTypeHandlingRunBook<WasCluster> implements RunBook {

	public WasClusterRunBook() {
		super(WasCluster.class);
	}

	public void resolve(Change<WasCluster> change, ChangePlan changePlan, List<Step> steps) {
		// applies only when Cluster is not identical
		WasCluster oldCluster = change.getOldRevision();
		WasCluster newCluster = change.getNewRevision();
		boolean identical = ConfigurationItemReflectionUtils.isIdentical(oldCluster, newCluster);
		if (!identical && !changeWillBeHandledByWasDeploymentRunBook(change, changePlan)) {

			if (change.isModification()) {
				getStepsForModifiedCluster(steps, oldCluster, newCluster, changePlan);
			}

			if (change.isDeletion()) {
				WasCluster cluster = change.getOldRevision();
				steps.add(new StopWasClusterStep(cluster, false));
				steps.add(new DestroyWasClusterStep(cluster));
			}

			if (change.isAddition()) {
				WasCluster cluster = change.getNewRevision();
				steps.add(new CreateWasClusterStep(cluster));
				for (WasManagedServer eachServer : cluster.getServers()) {
					steps.add(new AddSingleWasClusterMemberStep(cluster, eachServer));
					steps.add(new SynchronizeWasNodeStep(eachServer.getNode()));
					steps.add(new StartWasManagedServerStep(eachServer));
				}
			}
		}
	}

	@SuppressWarnings("unchecked")
	public void getStepsForModifiedCluster(List<Step> steps, WasCluster oldCluster, WasCluster newCluster, ChangePlan changePlan) {
		steps.add(new StopWasClusterStep(oldCluster, false));
		Set<WasNodeAgent> nodesToSynchronize = new HashSet<WasNodeAgent>();
		
		Collection<WasManagedServer> serversToBeRemoved = CollectionUtils.subtract(oldCluster.getServers(), newCluster.getServers());
		for (WasManagedServer serverToBeRemoved : serversToBeRemoved) {
			steps.add(new RemoveWasClusterMemberStep(oldCluster, serverToBeRemoved));
			nodesToSynchronize.add(serverToBeRemoved.getNode());
		}
		
		Collection<Change<WasManagedServer>> modifiedServerChanges = findWasServersToBeModified(changePlan, newCluster);
		for (Change<WasManagedServer> modifiedServerChange : modifiedServerChanges) {
			steps.add(new ModifyWasServerStep(modifiedServerChange.getNewRevision()));
		}
		
		Collection<WasManagedServer> serversToBeAdded = CollectionUtils.subtract(newCluster.getServers(), oldCluster.getServers());
		for (WasManagedServer serverToBeAdded : serversToBeAdded) {
			steps.add(new AddSingleWasClusterMemberStep(oldCluster, serverToBeAdded));
			nodesToSynchronize.add(serverToBeAdded.getNode());
		}
		
		for (WasNodeAgent node : nodesToSynchronize) {
			steps.add(new SynchronizeWasNodeStep(node));
		}
		
		for (WasManagedServer serverToBeStarted : newCluster.getServers()) {
			steps.add(new StartWasManagedServerStep(serverToBeStarted));
		}
		
	}

	@SuppressWarnings("unchecked")
	private Collection<Change<WasManagedServer>> findWasServersToBeModified(ChangePlan changePlan, WasCluster newCluster) {
		Collection<Change<WasManagedServer>> wasServersToBeModified = new ArrayList<Change<WasManagedServer>>();
		for (Change<?> change : changePlan.getChanges()) {
			if (change.getConfigurationItemClass() == WasManagedServer.class && change.isModification()) {
				if (newCluster.getServers().contains(change.getNewRevision())) {
					wasServersToBeModified.add( (Change<WasManagedServer>) change);
				}
			}
		}
		return wasServersToBeModified;
	}

	private boolean changeWillBeHandledByWasDeploymentRunBook(Change<WasCluster> clusterChange, ChangePlan changePlan) {
		Change<Deployment> deploymentChange = findChangeForDeploymentToClusterChange(clusterChange, changePlan);
		return (deploymentChange != null);
	}

	@SuppressWarnings("unchecked")
	private Change<Deployment> findChangeForDeploymentToClusterChange(Change<WasCluster> clusterChange, ChangePlan changePlan) {
		for (Change<?> change : changePlan.getChanges()) {
			if (change.getConfigurationItemClass() == Deployment.class) {
				WasCluster oldCluster = clusterChange.getOldRevision();
				WasCluster newCluster = clusterChange.getNewRevision();
				Deployment oldDeployment = (Deployment) change.getOldRevision();
				Deployment newDeployment = (Deployment) change.getNewRevision();
				
				if (oldDeployment != null) {
					if (oldCluster != null && oldDeployment.getTarget().getMembersOfType(WasCluster.class).contains(oldCluster)) {
						return (Change<Deployment>) change;
					} else if (newCluster != null && oldDeployment.getTarget().getMembersOfType(WasCluster.class).contains(newCluster)) {
						return (Change<Deployment>) change;
					}
				}
				
				if (newDeployment != null) {
					if (oldCluster != null && newDeployment.getTarget().getMembersOfType(WasCluster.class).contains(oldCluster)) {
						return (Change<Deployment>) change;
					} else if (newCluster != null && newDeployment.getTarget().getMembersOfType(WasCluster.class).contains(newCluster)) {
						return (Change<Deployment>) change;
					}
				}
			}
		}
		return null;
	}
}
