package com.xebialabs.deployit.plugin.was.container;

import static com.google.common.collect.Maps.newHashMap;
import static org.apache.commons.lang.StringUtils.isEmpty;
import static org.apache.commons.lang.StringUtils.isNotBlank;

import java.util.Map;

import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.inspection.InspectionProperty;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.PythonInspectionStep;
import com.xebialabs.deployit.plugin.python.PythonManagedContainer;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;
import com.xebialabs.deployit.plugin.was.step.CheckWasInstallationStep;
import com.xebialabs.deployit.plugin.was.step.GetWasVersionStep;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.OverthereFile;

@SuppressWarnings("serial")
@Metadata(description = "Base class for a Webshpere Cell")
public abstract class BaseCell extends BaseContainer implements Cell, PythonManagedContainer, PythonManagingContainer {

	@Property(required = true, asContainment = true, description = "Host on which the unmanaged WAS server runs")
	@InspectionProperty(required = true)
	private Host host;

	@Property(required = true, label = "WebSphere Installation Path", description = "Root path of the WebSphere installation path. e.g. /opt/ws/6.1/appserver/profiles/AppSrv01", size = Property.Size.LARGE)
	@InspectionProperty(required = true)
	private String wasHome;

	@Property(required = false, label = "Administrative port", description = "TCP port which is used to login to the WebSphere Administration, default is 8880")
	@InspectionProperty
	private int port;

	@Property(required = false, label = "Administrative username", description = "Username which is used to login to the WebSphere Administration.")
	@InspectionProperty
	private String username;

	@Property(required = false, password = true, label = "Administrative password", description = "Password which is used to login to the WebSphere Administration.")
	@InspectionProperty
	private String password;

	@Property(required = true, label = "WAS version", description = "Version of WebSphere Application Server.")
	private WasVersion version;

	@Property(required = true, label = "Run with daemon", description = "Set to true to execute commands with the Python daemon", defaultValue = "true", hidden = true)
	private boolean runWithDaemon;
	
	private static final String CONFIG_DIR_NAME = "config";

	private static final String PLUGIN_FILENAME = "plugin-cfg.xml";

	@Override
	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	@Override
	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	@Override
	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	@Override
	public String getWasHome() {
		return wasHome;
	}

	public void setWasHome(String wasHome) {
		this.wasHome = wasHome;
	}

	@Override
	public WasVersion getVersion() {
		return version;
	}

	public void setVersion(WasVersion version) {
		this.version = version;
	}

	@Override
	public Cell getCell() {
		return this;
	}

	@Override
	public Host getCellHost() {
		return host;
	}

	@Override
	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public String getConfigDirPath() {
		String fileSep = getCellHost().getOs().getFileSeparator();
		return getWasHome() + fileSep + CONFIG_DIR_NAME;
	}

	public String getPluginFilePath() {
		String fileSep = getCellHost().getOs().getFileSeparator();
		return getConfigDirPath() + fileSep + "cells" + fileSep + PLUGIN_FILENAME;
	}

	public String getWsadminPath() {
		return getCommandPath("wsadmin");
	}

	public String getVersionInfoPath() {
		return getCommandPath("versionInfo");
	}

	public String getStartServerPath() {
		return getCommandPath("startServer");
	}

	public String getStopServerPath() {
		return getCommandPath("stopServer");
	}

	public String getServerStatusPath() {
		return getCommandPath("serverStatus");
	}

	private String getCommandPath(String command) {
		String fileSep = getCellHost().getOs().getFileSeparator();
		String scriptExt = getCellHost().getOs().getScriptExtension();
		return getWasHome() + fileSep + "bin" + fileSep + command + scriptExt;
	}

	public boolean runWithDaemon() {
    	return runWithDaemon;
    }

	public void setRunWithDaemon(boolean runWithDaemon) {
    	this.runWithDaemon = runWithDaemon;
    }


	@Override
	public PythonManagingContainer getManagingContainer() {
		return this;
	}

	@Override
	public CmdLine getScriptCommandLine(OverthereFile pyFile) {
		CmdLine commandLine = new CmdLine();
		commandLine.addArgument(getWsadminPath());
		if (isNotBlank(getUsername())) {
			commandLine.addArgument("-user");
			commandLine.addArgument(getUsername());
			if (!isEmpty(getPassword())) {
				commandLine.addArgument("-password");
				commandLine.addPassword(getPassword());
			}
		}
		if (getPort() != 0) {
			commandLine.addArgument("-port");
			commandLine.addArgument(Integer.toString(getPort()));
		}
		commandLine.addArgument("-lang");
		commandLine.addArgument("jython");
		commandLine.addArgument("-f");
		commandLine.addArgument(pyFile.getPath());
		return commandLine;
	}

	@Override
	public String getRuntimePath() {
		return "was/runtime";
	}

	@Inspect
	public void inspect(InspectionPlanningContext ctx) {
		ctx.addStep(new CheckWasInstallationStep(this));
		ctx.addStep(new GetWasVersionStep(this));

		Map<String, Object> pythonVars = newHashMap();
		pythonVars.put("container", this);
		ctx.addStep(new PythonInspectionStep(this, this, "was/container/inspect-cell.py", pythonVars, "Inspect " + this));
	}

}
