package com.xebialabs.deployit.plugin.was.upgrade;

import java.util.Calendar;
import java.util.List;
import javax.jcr.Node;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.NodeType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.was.deployed.session.SessionManagerScope;
import com.xebialabs.deployit.server.api.repository.RawRepository;
import com.xebialabs.deployit.server.api.upgrade.Upgrade;
import com.xebialabs.deployit.server.api.upgrade.UpgradeException;
import com.xebialabs.deployit.server.api.upgrade.Version;

public class WAS410ModuleMigration extends Upgrade {

    public static final List<String> TYPES_TO_UPGRADE = Lists.newArrayList("was.Ear", "was.EarModule", "was.War", "was.WarModule");

    @Override
    public boolean doUpgrade(RawRepository repository) throws UpgradeException {
        logger.info("*** Starting WAS plugin 4.1.0 upgrade -- Migrate Module type ***");

        upgradeApplicationModules(repository);

        logger.info("*** Done WAS plugin 4.1.0 upgrade -- Migrate Module type ***");
        return true;
    }

    private void upgradeApplicationModules(final RawRepository repository) {
        for (String typeToUpgrade : TYPES_TO_UPGRADE) {
            List<Node> moduleNodes = repository.findNodesByType(Type.valueOf(typeToUpgrade));
            try {
                for (Node moduleNode : moduleNodes) {
                    if (!moduleNode.hasProperty("webModuleSessionTimeout")) continue;
                    logger.info("Migrating Module: {}", moduleNode.getPath());
                    createSessionManager(repository, moduleNode);
                }

            } catch (RepositoryException e) {
                throw new UpgradeException("*** Failed WAS plugin 4.1.0 upgrade -- Migrate Module type ***", e);
            }
        }
    }

    private void createSessionManager(RawRepository repository, Node moduleNode) throws RepositoryException {
        logger.info("Creating embedded was.SessionManager at [{}/sessionManager]", moduleNode.getPath().substring(1));

        Node sessionManagerNode = moduleNode.addNode("sessionManager");
        sessionManagerNode.addMixin("deployit:configurationItem");
        sessionManagerNode.addMixin(NodeType.MIX_VERSIONABLE);
        if(moduleNode.hasProperty("deployable")) {
            sessionManagerNode.setProperty("$configuration.item.type", "was.SessionManager");
            Node deployableModuleNode = moduleNode.getProperty("deployable").getNode();
            if(!deployableModuleNode.hasNode("sessionManager")) {
                createSessionManager(repository, deployableModuleNode);
            }
            sessionManagerNode.setProperty("deployable", deployableModuleNode.getNode("sessionManager"));
        }
        else {
            sessionManagerNode.setProperty("$configuration.item.type", "was.SessionManagerSpec");
        }
        if (moduleNode.hasProperty("webModuleSessionTimeout")) {
            sessionManagerNode.setProperty("TuningParams_invalidationTimeout", moduleNode.getProperty("webModuleSessionTimeout").getString());
        }
        sessionManagerNode.setProperty("$lastModified", Calendar.getInstance());
    }

    @Override
    public Version upgradeVersion() {
        return Version.valueOf("was-plugin","4.1.0");
    }

    private static final Logger logger = LoggerFactory.getLogger(WAS410ModuleMigration.class);
}
