/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.ci;

import com.xebialabs.deployit.*;
import com.xebialabs.deployit.ConfigurationItemProperty.Size;
import com.xebialabs.deployit.ci.Host;
import com.xebialabs.deployit.ci.OperatingSystemFamily;
import com.xebialabs.deployit.hostsession.*;
import com.xebialabs.deployit.plugin.wls.step.*;
import com.xebialabs.deployit.step.CheckDirExistsStep;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;

import java.util.*;

/**
 * A domain in a WebLogic WLS installation.
 * 
 * @see WlsCluster
 * @see WlsServer
 */
@SuppressWarnings("serial")
@ConfigurationItem(root = ConfigurationItemRoot.INFRASTRUCTURE, category = "middleware", discoverable = true, description = "A WebLogic Domain.")
public class WlsDomain extends BaseConfigurationItem implements Discoverable<WlsDomain> {

	private static final String WLS_SERVER_KEY = "WlsServer";

	private static final String WLS_CLUSTER_KEY = "WlsCluster";

	protected static final String GET_OBJECT_BY_CONTAINMENT_PATH_SCRIPT = WlsStepBase.STEP_RESOURCES_PATH + "get-object-by-containment-path.py";

	@ConfigurationItemProperty(required = true, identifying = true, description = "Name of the WebLogic Domain")
	private String name;

	@ConfigurationItemProperty(required = true, label = "WebLogic home", description = "The location of the WebLogic Server installation", size = Size.LARGE, discoveryParam = true)
	private String wlHome;

	@ConfigurationItemProperty(required = false, label = "WebLogic domain home", description = "The location of the WebLogic domain. Defaults to '<WebLogicHome>/../user_projects/domains/<Name>'", size = Size.LARGE)
	private String domainHome;

	@ConfigurationItemProperty(required = true, label = "Administrative server port", description = "Port to be used by the AdminServer for this domain", size = Size.SMALL, discoveryParam = true)
	private int port;

	@ConfigurationItemProperty(required = true, label = "Administrative username", description = "Username which is used to login to the WebLogic Domain.", discoveryParam = true)
	private String username;

	@ConfigurationItemProperty(required = true, label = "Administrative password", password = true, description = "Password which is used to login to the WebLogic Domain.", discoveryParam = true)
	private String password;

	@ConfigurationItemProperty(required = true, description = "The name of the admin server")
	private String adminServerName;

	@ConfigurationItemProperty(required = true, label = "Host", description = "The host that runs the admin server")
	private Host activeHost;

	@ConfigurationItemProperty(required = true, label = "Start Mode", description = "Tells how a managed server is start and stop, default is NodeManager, others are Script or Windows Service")
	private WlsStartMode startMode = WlsStartMode.NodeManager;

	@ConfigurationItemProperty(label = "Enable wlst.sh workaround", description = "Enable workaround for broken wlst.sh script found in some versions of WLS")
	private boolean enableWlstShWorkaround;

	@ConfigurationItemProperty
	private WlsVersion wlsVersion = WlsVersion.WEBLOGIC_10;

	public HostSession connectToAdminHost() {
		return HostSessionFactory.getHostSession(activeHost);
	}

	public int executeWlstJythonScript(CommandExecutionCallbackHandler handler, HostSession s, String scriptPath, CmdLine cmdLine) {
        CmdLine newCmdLine = new CmdLine();

		if (isEnableWlstShWorkaround() && getActiveHost().getOperatingSystemFamily() == OperatingSystemFamily.UNIX) {
			logger.info("Workaround for broken wlst.sh is enabled for " + this + ". Uploading wlst.sh from Deployit distribution");
			HostFile uploadedWlstSh = s.copyToTemporaryFile(new ClassPathResource("com/xebialabs/deployit/plugin/wls/wlst.sh"));
			newCmdLine.addArgument("/bin/sh");
			newCmdLine.addArgument(uploadedWlstSh.getPath());
			newCmdLine.addArgument(getWlHome());
		} else {
			newCmdLine.addArgument(getWlstPath());
		}

		newCmdLine.addArgument("-i");
		newCmdLine.addArgument(scriptPath);
        for (CmdLineArgument cmdLineArgument : cmdLine.getArguments()) {
            newCmdLine.add(cmdLineArgument);
        }

		return s.execute(handler, newCmdLine);
	}

	public String getWlstPath() {
		String fileSeparator = this.getActiveHost().getFileSeparator();
		String wlstExtension;
		if (getActiveHost().getOperatingSystemFamily() == OperatingSystemFamily.WINDOWS) {
			wlstExtension = ".cmd";
		} else {
			wlstExtension = getActiveHost().getScriptExtension();
		}
		return wlHome + fileSeparator + "common" + fileSeparator + "bin" + fileSeparator + "wlst" + wlstExtension;
	}

	public String getConnectUrl() {
		return activeHost.getAddress() + ":" + port;
	}

	public Host getActiveHost() {
		return activeHost;
	}

	public void setActiveHost(Host activeHost) {
		this.activeHost = activeHost;
	}

	public String getName() {
		return name;
	}

	public void setName(String name) {
		this.name = name;
	}

	public String getUsername() {
		return username;
	}

	public void setUsername(String username) {
		this.username = username;
	}

	public String getPassword() {
		return password;
	}

	public void setPassword(String password) {
		this.password = password;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getWlHome() {
		return wlHome;
	}

	public void setWlHome(String wlHome) {
		this.wlHome = wlHome;
	}

	public String getDomainHome() {
		return StringUtils.defaultIfEmpty(domainHome, getDefaultDomainHome());
	}

	private String getDefaultDomainHome() {
		final String fileSeparator = activeHost.getOperatingSystemFamily().getFileSeparator();
		return wlHome.substring(0, wlHome.lastIndexOf(fileSeparator)) + fileSeparator + "user_projects" + fileSeparator + "domains" + fileSeparator + name;
	}

	public void setDomainHome(String domainHome) {
		this.domainHome = domainHome;
	}

	public String getAdminServerName() {
		return adminServerName;
	}

	public void setAdminServerName(String adminServerName) {
		this.adminServerName = adminServerName;
	}

	public boolean isUseNodeManager() {
		return startMode == WlsStartMode.NodeManager;
	}

	public void setUseNodeManager(boolean useNodeManager) {
		this.startMode = WlsStartMode.NodeManager;
	}

	public WlsStartMode getStartMode() {
		return startMode;
	}

	public void setStartMode(WlsStartMode startMode) {
		this.startMode = startMode;
	}

	public boolean isEnableWlstShWorkaround() {
		return enableWlstShWorkaround;
	}

	public void setEnableWlstShWorkaround(boolean enableWlstShWorkaround) {
		this.enableWlstShWorkaround = enableWlstShWorkaround;
	}

	private List<Step> getCheckSteps() {
		List<Step> steps = new ArrayList<Step>();
		steps.add(new CheckDirExistsStep(this.getActiveHost(), this.getWlHome()));
		steps.add(new DetectBrokenWlstStep(this));
		steps.add(new RetrieveWlsDomainStep(this));
		steps.add(new RetrieveWlsSubtypesStep(this));
		return steps;
	}

	public Map<String, String> getObjectInfo(final String containmentpath) {
		WlstObjectInfoCollectingCallbackHandler handler = new WlstObjectInfoCollectingCallbackHandler();
		return getObjectInfoWithHandler(containmentpath, handler);
	}

	public Collection<String> getObjectInfoWithSubLevelByTypeNamesOnly(StepExecutionContext ctx, final String containmentpath) {
		List<String> names = new ArrayList<String>();
		WlstDirectoryObjectInfoCollectingCallbackHandler handler = new WlstDirectoryObjectInfoCollectingCallbackHandler();
		Map<String, String> objectInfo = getObjectInfoWithHandler(containmentpath, handler);
		for (String key : objectInfo.keySet()) {
			String name = objectInfo.get(key);
			names.add(name);
		}
		return names;
	}

	public Map<String, String> getDirectoryObjectInfo(final String containmentpath) {
		WlstDirectoryObjectInfoCollectingCallbackHandler handler = new WlstDirectoryObjectInfoCollectingCallbackHandler();
		return getObjectInfoWithHandler(containmentpath, handler);
	}

	private Map<String, String> getObjectInfoWithHandler(final String containmentpath, WlstObjectInfoCollectingCallbackHandlerBase handler) {
		HostSession s = connectToAdminHost();
		try {
			String scriptResourcePath = GET_OBJECT_BY_CONTAINMENT_PATH_SCRIPT;
			String scriptPath = s.copyToTemporaryFile(new ClassPathResource(scriptResourcePath)).getPath();
            CmdLine cmdLine = new CmdLine().addArgument(getUsername()).addPassword(getPassword()).addArgument(getConnectUrl()).addArgument(containmentpath);
            int res = executeWlstJythonScript(handler, s, scriptPath, cmdLine);
			if (res != 0) {
				return null;
			} else {
				return handler.getObjectInfo();
			}
		} finally {
			s.close();
		}
	}

	@SuppressWarnings("unchecked")
	public WlsDomain discoverChildrenInfo(Map<String, Object> discoveredAdditionalInfo, ChangePlan cp) {
		List<WlsServer> discoveredServers = new ArrayList<WlsServer>();
		if (discoveredAdditionalInfo.containsKey(WLS_SERVER_KEY)) {
			for (String serverName : (Collection<String>) discoveredAdditionalInfo.get(WLS_SERVER_KEY)) {
				if (serverName.equals(getAdminServerName())) {
					continue;
				}

				WlsServer wlsServer = new WlsServer();
				wlsServer.setDomain(this);
				wlsServer.setHost(this.getActiveHost()); // TODO Remove setting of the host here!!! retrieve from
				// 'Machine' property in Wls
				wlsServer.setName(serverName);
				wlsServer.setLabel("Server " + serverName + " on " + this.getName());
				wlsServer = wlsServer.discover(discoveredAdditionalInfo, cp);
				discoveredServers.add(wlsServer);
			}
		}
		discoveredAdditionalInfo.put("1_com.xebialabs.deployit.plugin.wls.ci.WlsServer", discoveredServers);

		List<WlsCluster> discoveredClusters = new ArrayList<WlsCluster>();
		if (discoveredAdditionalInfo.containsKey(WLS_CLUSTER_KEY)) {
			for (String clusterName : (Collection<String>) discoveredAdditionalInfo.get(WLS_CLUSTER_KEY)) {
				WlsCluster wlsCluster = new WlsCluster();
				wlsCluster.setDomain(this);
				wlsCluster.setName(clusterName);
				wlsCluster.setLabel("Cluster " + clusterName + " on " + this.getName());
				wlsCluster.setServers(getServersForCluster(discoveredAdditionalInfo, cp, wlsCluster, discoveredServers));
				wlsCluster = wlsCluster.discover(discoveredAdditionalInfo, cp);
				discoveredClusters.add(wlsCluster);
			}
		}
		discoveredAdditionalInfo.put("2_com.xebialabs.deployit.plugin.wls.ci.WlsCluster", discoveredClusters);

		return this;
	}

	private Set<WlsServer> getServersForCluster(Map<String, Object> discoveredAdditionalInfo, ChangePlan cp, WlsCluster wlsCluster,
	        List<WlsServer> discoveredServers) {
		Set<WlsServer> serversOnCluster = new HashSet<WlsServer>();
		for (WlsServer server : discoveredServers) {
			cp.addSteps(Collections.singletonList((Step) new CheckWlsServerIsOnCluster(wlsCluster, server)));
			cp.execute(discoveredAdditionalInfo);
			if (discoveredAdditionalInfo.get(server + CheckWlsServerIsOnCluster.SERVER_CLUSTER_SEPERATOR + wlsCluster) == CheckWlsServerIsOnCluster.IS_ON_RESULT) {
				serversOnCluster.add(server);
			}
		}
		return serversOnCluster;
	}

	public WlsDomain discover(Map<String, Object> info, ChangePlan cp) {
		cp.addSteps(getCheckSteps());
		cp.execute(info);
		return this;
	}

	public boolean isRunningVersion8() {
		return wlsVersion == WlsVersion.WEBLOGIC_8;
	}

	public WlsVersion getWlsVersion() {
		return wlsVersion;
	}

	public void setWlsVersion(WlsVersion wlsVersion) {
		this.wlsVersion = wlsVersion;
	}

	public boolean isRunningVersionGreaterThan9() {
		return wlsVersion != WlsVersion.WEBLOGIC_8;
	}

    private static final Logger logger = LoggerFactory.getLogger(WlsDomain.class);
}
