package com.xebialabs.deployit.plugin.wls.validator;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.execution.Step;
import com.xebialabs.deployit.plugin.api.udm.Deployable;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.wls.container.Cluster;
import com.xebialabs.deployit.plugin.wls.container.JmsServer;
import com.xebialabs.deployit.plugin.wls.container.JmsTarget;
import com.xebialabs.deployit.plugin.wls.deployed.AbstractQueue;
import com.xebialabs.deployit.plugin.wls.deployed.AbstractUniformDistributedQueue;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.deployit.plugin.wls.deployed.JmsResource;
import com.xebialabs.deployit.plugin.wls.resource.JmsResourceSpec;

import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.CREATE;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.MODIFY;

public class PlanLevelValidator {

	@PrePlanProcessor
	public Step<?> validatePlan(DeltaSpecification spec) {
		validateArtifactDeploymentOptions(spec);
		validateUniqueJNDIName(spec);
		validateUniformDistributedQueues(spec);
		validateQueueIsTargetedToSingleJmsServer(spec);
		return null;
	}

	private void validateArtifactDeploymentOptions(DeltaSpecification spec) {
		Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> deployeds = getAllDeployedArtifacts(spec);

		for (Deployable eachArtifact : deployeds.keySet()) {
			Collection<ExtensibleDeployedArtifact<DeployableArtifact>> artifactDeployeds = deployeds.get(eachArtifact);
			validateDeploymentOptionsAreSame(artifactDeployeds);
		}

	}

	@SuppressWarnings("unchecked")
	private Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> getAllDeployedArtifacts(DeltaSpecification spec) {
		Multimap<Deployable, ExtensibleDeployedArtifact<DeployableArtifact>> deployeds = HashMultimap.create();
		for (Delta each : spec.getDeltas()) {
			if ((each.getOperation() == CREATE || each.getOperation() == MODIFY)
			        && ExtensibleDeployedArtifact.class.isAssignableFrom(each.getDeployed().getClass())) {
				ExtensibleDeployedArtifact<DeployableArtifact> artifact = (ExtensibleDeployedArtifact<DeployableArtifact>) each.getDeployed();
				deployeds.put(artifact.getDeployable(), artifact);
			}
		}
		return deployeds;
	}

	private void validateDeploymentOptionsAreSame(Collection<ExtensibleDeployedArtifact<DeployableArtifact>> artifactDeployeds) {
		Iterator<ExtensibleDeployedArtifact<DeployableArtifact>> iterator = artifactDeployeds.iterator();
		if (iterator.hasNext()) {
			ExtensibleDeployedArtifact<DeployableArtifact> reference = iterator.next();

			while (iterator.hasNext()) {
				ExtensibleDeployedArtifact<DeployableArtifact> next = iterator.next();
				validateSameValues(reference, next);
			}
		}
	}

	private void validateSameValues(ExtensibleDeployedArtifact<DeployableArtifact> reference, ExtensibleDeployedArtifact<DeployableArtifact> deployed) {
		if (reference.getContainer().getDomain().equals(deployed.getContainer().getDomain())) {
			if (reference.isVersioned() != deployed.isVersioned()) {
				throw new RuntimeException("Can't have different values for 'versioned' property for same artifact " + reference.getDeployable().getName());
			}
			if (reference.getStageMode() != deployed.getStageMode()) {
				throw new RuntimeException("Can't have different values for 'stageMode' property for same artifact " + reference.getDeployable().getName());
			}
			if (reference.getStagingDirectory() != null && !reference.getStagingDirectory().equals(deployed.getStagingDirectory())) {
				throw new RuntimeException("Can't have different values for 'stagingDirectory' property for same artifact "
				        + reference.getDeployable().getName());
			}
			if (reference.getRedeploymentStrategy() != deployed.getRedeploymentStrategy()) {
				throw new RuntimeException("Can't have different values for the deployment strategy for the same artifact "
				        + reference.getDeployable().getName());
			}
		}
	}

	@SuppressWarnings({ "unchecked" })
	private void validateUniqueJNDIName(DeltaSpecification spec) {
		Map<String, String> jndiNames = new HashMap<String, String>();
		for (Delta each : spec.getDeltas()) {
			if ((each.getOperation() == CREATE || each.getOperation() == MODIFY) && JmsResource.class.isAssignableFrom(each.getDeployed().getClass())) {
				JmsResource<JmsResourceSpec, JmsTarget> jmsResource = (JmsResource<JmsResourceSpec, JmsTarget>) each.getDeployed();
				if (jndiNames.get(jmsResource.getJndiName()) != null && !jndiNames.get(jmsResource.getJndiName()).equals(jmsResource.getDeployable().getName())) {
					throw new RuntimeException(String.format("Jndi name %s is in use by multiple resource sources", jmsResource.getJndiName()));
				} else {
					jndiNames.put(jmsResource.getJndiName(), jmsResource.getDeployable().getName());
				}
			}
		}
	}

	private void validateUniformDistributedQueues(DeltaSpecification spec) {
		// A UDD must be targeted to servers within a single cluster or a single
		// stand-alone server
		Map<String, Cluster> uddQueueContainers = new HashMap<String, Cluster>();
		for (Delta each : spec.getDeltas()) {
			if ((each.getOperation() == CREATE || each.getOperation() == MODIFY)
			        && AbstractUniformDistributedQueue.class.isAssignableFrom(each.getDeployed().getClass())) {
				AbstractUniformDistributedQueue jmsResource = (AbstractUniformDistributedQueue) each.getDeployed();
				Cluster cluster = getContainerCluster(jmsResource);
				// the cluster to which the container belongs or null if it points to a stand-alone server
				if (uddQueueContainers.containsKey(jmsResource.getName()) && !uddQueueContainers.get(jmsResource.getName()).equals(cluster)) {
					throw new RuntimeException(String.format("The UDD %s must be targeted to servers within a single cluster or a single stand-alone server",
					        jmsResource.getName()));
				} else {
					uddQueueContainers.put(jmsResource.getName(), cluster);
				}
			}
		}
	}

	private void validateQueueIsTargetedToSingleJmsServer(DeltaSpecification spec) {
		Map<String, JmsServer> queueJmsServers = new HashMap<String, JmsServer>();
		for (Delta each : spec.getDeltas()) {
			if ((each.getOperation() == CREATE || each.getOperation() == MODIFY) && AbstractQueue.class.isAssignableFrom(each.getDeployed().getClass())) {
				AbstractQueue queue = (AbstractQueue) each.getDeployed();
				if (queueJmsServers.containsKey(queue.getName()) && !queue.getContainer().equals(queueJmsServers.get(queue.getName()))) {
					throw new RuntimeException((String.format("Queue %s must be targeted to only a single JmsServer", queue.getName())));
				} else {
					queueJmsServers.put(queue.getName(), queue.getContainer());
				}
			}
		}
	}

	private Cluster getContainerCluster(AbstractUniformDistributedQueue jmsResource) {
		JmsTarget container = jmsResource.getContainer();
		Cluster cluster = null;
		if (Cluster.class.isAssignableFrom(container.getClass())) {
			cluster = (Cluster) container;
		} else if (JmsServer.class.isAssignableFrom(container.getClass())) {
			cluster = ((JmsServer) container).getServer().getClusterIfAny();
		}
		return cluster;
	}

}
