/*
 * Copyright (c) 2008-2011 XebiaLabs B.V. All rights reserved.
 *
 * Your use of XebiaLabs Software and Documentation is subject to the Personal
 * License Agreement.
 *
 * http://www.xebialabs.com/deployit-personal-edition-license-agreement
 *
 * You are granted a personal license (i) to use the Software for your own
 * personal purposes which may be used in a production environment and/or (ii)
 * to use the Documentation to develop your own plugins to the Software.
 * "Documentation" means the how to's and instructions (instruction videos)
 * provided with the Software and/or available on the XebiaLabs website or other
 * websites as well as the provided API documentation, tutorial and access to
 * the source code of the XebiaLabs plugins. You agree not to (i) lease, rent
 * or sublicense the Software or Documentation to any third party, or otherwise
 * use it except as permitted in this agreement; (ii) reverse engineer,
 * decompile, disassemble, or otherwise attempt to determine source code or
 * protocols from the Software, and/or to (iii) copy the Software or
 * Documentation (which includes the source code of the XebiaLabs plugins). You
 * shall not create or attempt to create any derivative works from the Software
 * except and only to the extent permitted by law. You will preserve XebiaLabs'
 * copyright and legal notices on the Software and Documentation. XebiaLabs
 * retains all rights not expressly granted to You in the Personal License
 * Agreement.
 */

package com.xebialabs.deployit.plugin.wls.container;

import com.xebialabs.deployit.plugin.api.inspection.Inspect;
import com.xebialabs.deployit.plugin.api.inspection.InspectionPlanningContext;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.api.udm.base.BaseContainer;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.deployit.plugin.python.PythonInspectionStep;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;

import java.util.Collections;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Preconditions.checkState;
import static com.google.common.collect.Maps.newHashMap;

/**
 * A WLS server (Managed or admin) in a {@link Domain WLS domain}.
 *
 * @see Cluster
 * @see Domain
 */
@SuppressWarnings("serial")
@Metadata(description = "WebLogic Server that runs in its own Java Virtual Machine (JVM) and has its own configuration")
public class Server extends BaseContainer implements WlsContainer, JmsTarget {

	@Property(required = true, description = "Port for the server runs on")
	private int port;

	@Property(required = true, asContainment = true, description = "WebLogic domain to which this server belongs. 'asContainment'=true, means a Server is 'contained' under a Domain")
	private Domain domain;

	@Property(required = false, description = "Host on which this server is running, needed to perform no-stage deployments and to start the server via a script")
	private Host host;

	@Property(required = false, label = "Managed server start command", description = "Command that should be executed to start the managed server.")
	private String startCommand;

	@Property(required = false, label = "Managed server stop command", description = "Command that should be executed to stop the managed server.")
	private String stopCommand;

	@Property(required = false, description = "Environment variables for server", label = "Environment Variables")
	private Map<String, String> envVars = newHashMap();

	@Property(required = true, defaultValue = "80", hidden = true)
	private int startOrder;

	@Property(required = true, defaultValue = "20", hidden = true)
	private int stopOrder;

	@Override
	public String getObjectName() {
		return "com.bea:Name=" + getName() + ",Type=Server";
	}

	@Inspect
	public void inspect(InspectionPlanningContext ctx) {
		Map<String, Object> pythonVars = newHashMap();
		pythonVars.put("container", this);
		ctx.addStep(new PythonInspectionStep(this, this.getManagingContainer(), "wls/container/inspect-server.py", pythonVars, "Inspect " + this));
	}

	public Domain getDomain() {
		return domain;
	}

	@Override
	public Set<Host> getHosts() {
		checkState(host != null, "Can only perform no-stage deployments to server that has their host property set");
		return Collections.singleton(host);
	}

	/**
	 * find out the cluster to which this server belongs
	 *
	 * @return the Cluster to which the server belongs or null if it doesn't belong to any cluster
	 */
	public Cluster getClusterIfAny() {
		Set<Cluster> clusters = domain.getClusters();
		for (Cluster eachCluster : clusters) {
			if (eachCluster.getServers().contains(this)) {
				return eachCluster;
			}
		}
		return null;
	}

	@Override
	public Set<Server> getServers() {
		return Collections.singleton(this);
	}

	public void setDomain(Domain domain) {
		this.domain = domain;
	}

	public Host getHost() {
		return host;
	}

	public void setHost(Host host) {
		this.host = host;
	}

	public int getPort() {
		return port;
	}

	public void setPort(int port) {
		this.port = port;
	}

	public String getStartCommand() {
		return startCommand;
	}

	public void setStartCommand(String startCommand) {
		this.startCommand = startCommand;
	}

	public String getStopCommand() {
		return stopCommand;
	}

	public void setStopCommand(String stopCommand) {
		this.stopCommand = stopCommand;
	}

	@Override
	public PythonManagingContainer getManagingContainer() {
		return domain;
	}

	public int getStartOrder() {
		return startOrder;
	}

	public int getStopOrder() {
		return stopOrder;
	}
}
