package com.xebialabs.deployit.plugin.wls.contributor;

import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.xebialabs.deployit.plugin.api.deployment.execution.DeploymentStep;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.wls.container.JmsServer;
import com.xebialabs.deployit.plugin.wls.container.Server;
import com.xebialabs.deployit.plugin.wls.container.WlsContainer;
import com.xebialabs.deployit.plugin.wls.step.StartWlsServerWithScriptStep;
import com.xebialabs.deployit.plugin.wls.step.StartWlsTargetStep;
import com.xebialabs.deployit.plugin.wls.step.StopWlsTargetStep;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Predicates.*;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Maps.transformEntries;
import static com.xebialabs.deployit.plugin.api.deployment.specification.Operation.NOOP;
import static com.xebialabs.deployit.plugin.wls.contributor.Predicates2.*;

public class DeployedAndCopiedArtifactLifeCycleContributor {

    static final int SHIFT_RESTART_ORDERS = 1;

    @Contributor
    public void restartAfterDeployResourcesOrCopiedArtifact(Deltas deltas, DeploymentPlanningContext result) {

        final Set<Deployed<?, ?>> restartableDeployed = Sets.filter(gatherTargets(deltas.getDeltas()),
                new Predicate<Deployed<?, ?>>() {
                    @Override
                    public boolean apply(Deployed<?, ?> deployed) {
                        final boolean restartTarget = (Boolean) deployed.getProperty("restartTarget");
                        return restartTarget;
                    }
                });
        logger.debug("restartable deployeds {}", restartableDeployed);

        //assign the higher order for each targeted containers
        final Map<WlsContainer, Integer> ordersByContainer = Maps.newHashMap();
        for (Deployed<?, ?> target : restartableDeployed) {
            final int createOrder = (Integer) target.getProperty("createOrder");
            final WlsContainer container;
            if (target.getContainer() instanceof JmsServer) {
                JmsServer server = (JmsServer) target.getContainer();
                container = server.getServer();
            }
            else {
                container = (WlsContainer) target.getContainer();
            }

            if (!ordersByContainer.containsKey(container)) {
                ordersByContainer.put(container, 0);
            }
            if (createOrder > ordersByContainer.get(container)) {
                ordersByContainer.put(container, createOrder);
            }
        }

        //Transform the values of the map to list of steps.
        final Collection<List<DeploymentStep>> steps = transformEntries(ordersByContainer, new Maps.EntryTransformer<WlsContainer, Integer, List<DeploymentStep>>() {
            @Override
            public List<DeploymentStep> transformEntry(WlsContainer container, Integer order) {
                logger.debug("create stop/start step for {} --> {} ", container, order);
                final ImmutableList.Builder<DeploymentStep> builder = new ImmutableList.Builder<DeploymentStep>();
                builder.add(new StopWlsTargetStep(order + SHIFT_RESTART_ORDERS, container));
                for (Server s : container.getServers()) {
                    switch (s.getDomain().getStartMode()) {
                        case NodeManager:
                            builder.add(new StartWlsTargetStep(order + SHIFT_RESTART_ORDERS, s));
                            break;
                        case Script:
                            builder.add(new StartWlsServerWithScriptStep(order + SHIFT_RESTART_ORDERS, s));
                    }
                }
                return builder.build();
            }
        }).values();

        //Add all !
        for (List<DeploymentStep> step : steps) {
            result.addSteps(step);
        }
    }

    private Set<Deployed<?, ?>> gatherTargets(List<Delta> operations) {
        logger.error("DeployedAndCopiedArtifactLifeCycleContributor.gatherTargets");
        final Type resourcesType = DescriptorRegistry.getDescriptor("wls.Resource").getType();
        final Type copiedArtifactType = DescriptorRegistry.getDescriptor("wls.CopiedArtifact").getType();
        final Iterable<Delta> filter = filter(
                operations,
                and(not(operationIs(NOOP)),
                        or(deltaOf(resourcesType), deltaOf(copiedArtifactType))));
        final Iterable<Deployed<?, ?>> transform = Iterables.transform(filter, extractDeployed());
        final Set<Deployed<?, ?>> deployeds = Sets.newHashSet(transform);
        logger.debug("deployed {}", deployeds);
        return deployeds;
    }

    protected static final Logger logger = LoggerFactory.getLogger(DeployedAndCopiedArtifactLifeCycleContributor.class);

}
