package com.xebialabs.deployit.plugin.wls.preplanprocessor;

import java.io.File;
import java.io.IOException;
import java.util.jar.Attributes;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;

import com.google.common.base.Function;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;

import com.xebialabs.deployit.plugin.api.deployment.planning.PrePlanProcessor;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.DeltaSpecification;
import com.xebialabs.deployit.plugin.api.flow.Step;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;
import com.xebialabs.overthere.local.LocalFile;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Iterables.filter;
import static com.google.common.collect.Iterables.transform;
import static com.google.common.io.Closeables.closeQuietly;

public class WlsArtifactProcessor {

    @PrePlanProcessor
    public Step setVersionIdentifiers(DeltaSpecification spec) {

        final String appVersion = spec.getDeployedApplication().getVersion().getVersion();
        final String appName = spec.getDeployedApplication().getVersion().getApplication().getName();
        final String defaultVersion = appName + "-" + appVersion;

        final VersionSetter version = new VersionSetter(defaultVersion);
        for (ExtensibleDeployedArtifact<?> artifact : filter(transform(spec.getDeltas(), TO_DEPLOYED), IS_VERSIONED)) {
            if (isNullOrEmpty(artifact.getVersionIdentifier())) {
                version.apply(artifact);
            }
        }
        return null;
    }

    private static final Function<Delta, ExtensibleDeployedArtifact<?>> TO_DEPLOYED = new Function<Delta, ExtensibleDeployedArtifact<?>>() {
        @Override
        public ExtensibleDeployedArtifact<?> apply(Delta delta) {
            Deployed<?, ?> deployed = delta.getDeployed();
            return deployed instanceof ExtensibleDeployedArtifact ? (ExtensibleDeployedArtifact<?>) deployed : null;
        }
    };

    private static final Predicate<ExtensibleDeployedArtifact<?>> IS_VERSIONED = new Predicate<ExtensibleDeployedArtifact<?>>() {
        @Override
        public boolean apply(ExtensibleDeployedArtifact<?> input) {
            return Predicates.<Object>notNull().apply(input) && ( input.isVersioned() || isSharedLibrary(input));
        }

        private boolean isSharedLibrary(ExtensibleDeployedArtifact<?> input) {
            Type inputType = input.getType();
            return inputType.equals(ExtensibleDeployedArtifact.DEPLOYED_SHARED_LIB_WAR);
        }
    };


    private class VersionSetter implements Function<ExtensibleDeployedArtifact<?>, Void> {

        private final String defaultVersion;

        public VersionSetter(String defaultVersion) {
            this.defaultVersion = defaultVersion;
        }

        @Override
        public Void apply(ExtensibleDeployedArtifact<?> deployed) {
            final Artifact artifact = deployed.getDeployable();

            final File manifestFile = ((LocalFile) artifact.getFile()).getFile();
            final Manifest manifest = readManifest(manifestFile);
            if (manifest == null)
                return null;
            final Attributes mainAttributes = manifest.getMainAttributes();
            final String extName = mainAttributes.getValue("Extension-Name");

            deployed.setVersioned(true);
            if (isNullOrEmpty(extName)) {
                deployed.setVersionIdentifier(defaultVersion);
            } else {
                final String specVersion = mainAttributes.getValue("Specification-Version");
                final String implVersion = mainAttributes.getValue("Implementation-Version");
                deployed.setVersionIdentifier(isNullOrEmpty(implVersion) ? specVersion : specVersion + "@" + implVersion);
            }
            return null;
        }

        private Manifest readManifest(File archiveFile) {
            Manifest manifest;
            JarInputStream in = null;
            try {
                in = new JarInputStream(new java.io.FileInputStream(archiveFile));
                manifest = in.getManifest();
            } catch (IOException exception) {
                throw new RuntimeException("Cannot read manifest file " + archiveFile, exception);
            } finally {
                closeQuietly(in);
            }
            return manifest;
        }
    }



}
