package com.xebialabs.deployit.plugin.wls.step;

import com.google.common.base.Function;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.udm.DeployableArtifact;
import com.xebialabs.deployit.plugin.api.udm.Deployed;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.python.PythonDeploymentStep;
import com.xebialabs.deployit.plugin.python.PythonManagingContainer;
import com.xebialabs.deployit.plugin.wls.deployed.BaseWlsDeployed;
import com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact;

import java.util.Collection;
import java.util.Map;
import java.util.Set;

import static com.google.common.base.Joiner.on;
import static com.google.common.base.Predicates.notNull;
import static com.google.common.collect.Collections2.filter;
import static com.google.common.collect.Collections2.transform;
import static com.google.common.collect.ImmutableSet.copyOf;
import static com.google.common.collect.Iterables.getFirst;
import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Ordering.natural;
import static com.xebialabs.deployit.plugin.wls.deployed.ExtensibleDeployedArtifact.VAR_TARGETS;
import static java.lang.String.format;

public class MultiTargetDeploymentStep extends PythonDeploymentStep {
    private static final Function<StepData, String> GET_TARGET_CONTAINER_NAME = new Function<StepData, String>() {
        @Override
        public String apply(StepData target) {
            return target.getDeployedArtifact().getContainer().getName();
        }
    };
    private static final Function<StepData, Delta> GET_DELTA = new Function<StepData, Delta>() {
        @Override
        public Delta apply(StepData target) {
            return target.getDelta();
        }
    };

    private Collection<Delta> deltas;

    public MultiTargetDeploymentStep(int order, PythonManagingContainer container, String scriptPath, Map<String, Object> pythonVars, String description) {
        super(order, container, scriptPath, pythonVars, description);
    }

    public static MultiTargetDeploymentStep create(DeploymentPlanningContext ctx, Collection<StepData> singleSteps) {
        StepData singleStep = getFirst(singleSteps, null);
        Collection<String> targets = natural().sortedCopy(copyOf(transform(singleSteps, GET_TARGET_CONTAINER_NAME)).asList());

        Collection<Delta> deltas = newArrayList(filter(transform(singleSteps, GET_DELTA), notNull()));
        Delta delta = getFirst(deltas, null);
        Deployed<?, ?> previousDeployed = delta != null ? delta.getPrevious() : null;

        PythonManagingContainer container = getManagingContainer(singleStep.deployedArtifact);
        Map<String, Object> pythonVars = buildPythonVars(ctx.getDeployedApplication(), previousDeployed, singleStep.deployedArtifact, targets, singleStep.getOptions());
        String description = getDescription(singleStep.verb, singleStep.deployedArtifact, targets);

        MultiTargetDeploymentStep multiStep = new MultiTargetDeploymentStep(singleStep.order, container, singleStep.scriptPath, pythonVars, description);
        multiStep.setUploadArtifactData(singleStep.options.contains(BaseWlsDeployed.STEP_OPTION_UPLOAD_ARTIFACT_DATA));
        multiStep.setDeltas(deltas);
        return multiStep;
    }

    private static PythonManagingContainer getManagingContainer(ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact) {
        return deployedArtifact.getContainer().getManagingContainer();
    }

    private static Map<String, Object> buildPythonVars(DeployedApplication deployedApplication, Deployed<?, ?> previousDeployed, ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact, Collection<String> targets, Set<String> options) {
        Map<String, Object> pythonVars = deployedArtifact.getPythonVars(previousDeployed, deployedApplication, options);
        pythonVars.put(VAR_TARGETS, on(",").join(targets));
        return pythonVars;
    }

    private static String getDescription(String verb, ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact, Collection<String> targets) {
        return format("%s %s on %s: %s", verb, (deployedArtifact.getDeployable()).getFile().getName(), deployedArtifact.getContainer().getDomain().getName(), on(",").join(targets));
    }

    @Override
    public String getDescription() {
        return super.getDescription();
    }

    public void setDeltas(Collection<Delta> deltas) {
        this.deltas = deltas;
    }

    public Collection<Delta> getDeltas() {
        return deltas;
    }

    public static class StepData {
        private final ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact;
        private final int order;
        private final String scriptPath;
        private final String verb;
        private final Delta delta;
        private final Set<String> options;

        public StepData(ExtensibleDeployedArtifact<? extends DeployableArtifact> deployedArtifact, int order, String scriptPath, String verb, Set<String> options, Delta delta) {

            this.deployedArtifact = deployedArtifact;
            this.order = order;
            this.scriptPath = scriptPath;
            this.verb = verb;
            this.options = options;
            this.delta = delta;
        }

        public ExtensibleDeployedArtifact<? extends DeployableArtifact> getDeployedArtifact() {
            return deployedArtifact;
        }

        public int getOrder() {
            return order;
        }

        public String getScriptPath() {
            return scriptPath;
        }

        public String getVerb() {
            return verb;
        }

        public Set<String> getOptions() {
            return options;
        }

        public Delta getDelta() {
            return delta;
        }

        @Override
        public String toString() {
            final StringBuffer sb = new StringBuffer("Step{");
            sb.append("deployedArtifact=").append(deployedArtifact);
            sb.append(", order=").append(order);
            sb.append(", scriptPath='").append(scriptPath).append('\'');
            sb.append(", verb='").append(verb).append('\'');
            sb.append('}');
            return sb.toString();
        }
    }
}
