package com.crossingchannels.portal.websphere.ci.base.contributor;

import java.lang.reflect.ParameterizedType;

import org.apache.commons.lang.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.crossingchannels.portal.websphere.ci.base.util.CastHelper;
import com.crossingchannels.portal.websphere.ci.base.util.TypeHelper;
import com.crossingchannels.portal.websphere.util.InternalPluginException;
import com.xebialabs.deployit.plugin.api.deployment.planning.Contributor;
import com.xebialabs.deployit.plugin.api.deployment.planning.DeploymentPlanningContext;
import com.xebialabs.deployit.plugin.api.deployment.specification.Delta;
import com.xebialabs.deployit.plugin.api.deployment.specification.Deltas;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Deployed;

/**
 * Abstract contributor to add additional steps for (un)installation of portlet archives.
 * 
 * @author fwiegerinck
 * @since 1.1
 * @version 1.1
 */
public abstract class AbstractStepsContributor<D extends Deployed<?, ?>> {

    /**
     * Define logger for this contributor.
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(AbstractStepsContributor.class);

    /**
     * Store base CI type. Used to recognize CI type.
     */
    private final Type targetCiType;
    
    /**
     * Store container CI type. Used to recognize if target container matches criteria.
     */
    private final Type targetContainerType;
    
    /**
     * Store Java class used for template methods to perform actual install, update or uninstall of CI.
     */
    private final Class<D> baseJavaType;

    /**
     * Initialize new abstract contributor.
     * 
     * @param targetCiType
     *            Target CI type.
     * @param targetContainerType
     *            Target container type.
     */
	protected AbstractStepsContributor(final Type targetCiType, final Type targetContainerType) {
        super();

        Validate.notNull(targetCiType);
        Validate.notNull(targetContainerType);

        // Store information
        this.targetCiType = targetCiType;
        this.targetContainerType = targetContainerType;
        if ( getClass().getGenericSuperclass() instanceof ParameterizedType ) {
	        java.lang.reflect.Type type = ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
	        if ( type instanceof Class ) {
	        	this.baseJavaType = CastHelper.cast(type);
	        } else {
	        	this.baseJavaType = CastHelper.cast(((ParameterizedType) type).getRawType());
	        }
        } else {
        	this.baseJavaType = CastHelper.cast(Deployed.class);
        }
    }

    @Contributor
    public void contributePortletSteps(final Deltas deltas, final DeploymentPlanningContext context) {

        // Scan all deltas for WAR deployments
        for (final Delta delta : deltas.getDeltas()) {

            switch (delta.getOperation()) {
            case CREATE:
                this.processInstallationOfCi(delta, context);
                break;
            case MODIFY:
                this.processUpdateOfCi(delta, context);
                break;
            case DESTROY:
                this.processUninstallationOfCi(delta, context);
                break;
            case NOOP:
                break;
            default:
                throw new IllegalStateException("Unknown delta operation: " + delta.getOperation());
            }
        }

    }

    protected void processInstallationOfCi(final Delta delta, final DeploymentPlanningContext context) {
        // Preconditions
        Validate.notNull(delta);
        Validate.notNull(context);

        // Get deployed CI and ensure it is valid
        final Deployed<?, ?> rawDeployedCi = delta.getDeployed();

        // Log
        AbstractStepsContributor.LOGGER.trace("Called to handle install request for CI with id [{}]", rawDeployedCi.getId());

        // Determine whether or not this contributor should perform any action
        if (!this.matchesCiTypeCriteria(rawDeployedCi)) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.trace("CI [{}] doesn't match criteria, skip it...", rawDeployedCi.getId());
            return;
        }
        
        if (!this.baseJavaType.isInstance(rawDeployedCi) ) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.error("Unable to process CI [{}]. CI class [{}] doesn't match required (generics) type [{}]", new Object[]{rawDeployedCi.getId(),rawDeployedCi.getClass(), this.baseJavaType});
            throw new InternalPluginException("Found CI that matches CI type criteria (deployed and container) that doesn't match java template type (generics)");        	
        }

        // Perform actual call to install method
        callInstall(context, rawDeployedCi, delta);
    }

    protected void processUpdateOfCi(final Delta delta, final DeploymentPlanningContext context) {
        // Preconditions
        Validate.notNull(delta);
        Validate.notNull(context);

        // Get deployed CI and ensure it is valid
        final Deployed<?, ?> rawDeployedCi = delta.getDeployed();
        final Deployed<?, ?> rawPreviousCi = delta.getPrevious();

        // Log
        AbstractStepsContributor.LOGGER.trace("Called to handle update request for CI with id [{}]. Previous CI has id [{}]", rawDeployedCi.getId(), rawPreviousCi);

        // If both 'deployed' and 'previous' CI is valid, continue
        // If only 'deployed' CI is valid, handle as install
        // If only 'previous' CI is valid, handle as uninstall
        final boolean deployedCiValid = this.matchesCiTypeCriteria(rawDeployedCi);
        final boolean previousCiValid = this.matchesCiTypeCriteria(rawPreviousCi);
        if (deployedCiValid && previousCiValid) {

            // Only perform update if containers are the same
            if (rawDeployedCi.getContainer().equals(rawPreviousCi.getContainer())) {

                // Log
                AbstractStepsContributor.LOGGER.trace("Previous and target CI [{}] do have the same container. Perform an update.", rawDeployedCi.getId());

                // Perform update
                this.callUpdate(context, rawDeployedCi, rawPreviousCi, delta);
            } else {

                // Log
                AbstractStepsContributor.LOGGER.trace(
                        "Previous and target CI [{}] don't have the same container. Cannot perform an update for different containers. Perform an uninstall and install instead.",
                        rawDeployedCi.getId());

                // Not the same, perform uninstall and install
                this.callUninstall(context, rawPreviousCi, delta);
                this.callInstall(context, rawDeployedCi, delta);
            }

        } else if (deployedCiValid && !previousCiValid) {
            // Log
            AbstractStepsContributor.LOGGER.trace("Previous CI [{}] is not a valid portlet deployed archive. Perform an install only...", rawDeployedCi.getId());

            // Do call
            this.callInstall(context, rawDeployedCi, delta);
        } else if (!deployedCiValid && previousCiValid) {
            // Log
            AbstractStepsContributor.LOGGER.trace("Target CI [{}] is not a valid portlet deployed archive. Perform an uninstall only...", rawDeployedCi.getId());

            // Do call
            this.callUninstall(context, rawPreviousCi, delta);
        } // else skip
    }

    protected void processUninstallationOfCi(final Delta delta, final DeploymentPlanningContext context) {
        // Preconditions
        Validate.notNull(delta);
        Validate.notNull(context);

        // Get previous CI and ensure it is valid
        final Deployed<?, ?> rawPreviousCi = delta.getPrevious();

        // Log
        AbstractStepsContributor.LOGGER.trace("Called to handle uninstall request for CI with id [{}]", rawPreviousCi.getId());

        // Determine whether or not this contributor should perform any action
        if (!this.matchesCiTypeCriteria(rawPreviousCi)) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.trace("CI [{}] doesn't match criteria, skip it...", rawPreviousCi.getId());
            return;
        }

        // Do call
        this.callUninstall(context, rawPreviousCi, delta);
    }

    /**
     * Test whether or not the CI is valid.
     * 
     * @param ciToValidate
     *            The CI to validate.
     * @return TRUE if valid, FALSE otherwise.
     */
    protected boolean matchesCiTypeCriteria(final Deployed<?, ?> ciToValidate) {
        // Ensure not NULL
        Validate.notNull(ciToValidate, "CI to validate cannot be NULL");

        // Test if CI is valid (correct type and correct target container)
        return (TypeHelper.isInstanceOf(ciToValidate.getType(), this.targetCiType) && TypeHelper.isInstanceOf(ciToValidate.getContainer().getType(), this.targetContainerType));
    }
    
    /**
     * Perform type conversion and call install method.
     * 
     * @param context The deployment context used to perform install.
     * @param rawDeployedCi The raw CI type to install.
     * @param delta
     * 			  The delta currently processed.
     */
	protected final void callInstall(final DeploymentPlanningContext context,
			final Deployed<?, ?> rawDeployedCi, final Delta delta) {

        if (!this.baseJavaType.isInstance(rawDeployedCi) ) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.error("Unable to process CI [{}]. CI class [{}] doesn't match required (generics) type [{}]", new Object[]{rawDeployedCi.getId(),rawDeployedCi.getClass(), this.baseJavaType});
            throw new InternalPluginException("Found CI that matches CI type criteria (deployed and container) that doesn't match java template type (generics)");        	
        }
		
		// Cast objects
        final D typedDeployedCi = this.baseJavaType.cast(rawDeployedCi);

        // Log
        AbstractStepsContributor.LOGGER.debug("Add steps to install CI with id [{}]", rawDeployedCi.getId());

        // Do call
        this.install(typedDeployedCi, context, delta);
	}    

    /**
     * Perform type conversion and call update method.
     * 
     * @param context The deployment context used to perform update.
     * @param rawDeployedCi The raw CI type to install.
     * @param rawPreviousCi The raw CI type to uninstall.
     * @param delta
     * 			  The delta currently processed.
     */
	protected final void callUpdate(final DeploymentPlanningContext context,
			final Deployed<?, ?> rawDeployedCi, final Deployed<?, ?> rawPreviousCi, final Delta delta) {
		
        if (!this.baseJavaType.isInstance(rawDeployedCi) ) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.error("Unable to process CI [{}]. CI class [{}] doesn't match required (generics) type [{}]", new Object[]{rawDeployedCi.getId(),rawDeployedCi.getClass(), this.baseJavaType});
            throw new InternalPluginException("Found CI that matches CI type criteria (deployed and container) that doesn't match java template type (generics)");        	
        }
        if (!this.baseJavaType.isInstance(rawPreviousCi) ) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.error("Unable to process CI [{}]. CI class [{}] doesn't match required (generics) type [{}]", new Object[]{rawPreviousCi.getId(),rawPreviousCi.getClass(), this.baseJavaType});
            throw new InternalPluginException("Found CI that matches CI type criteria (previous deployed CI and container) that doesn't match java template type (generics)");        	
        }		

        // Cast objects
        final D typedDeployedCi = this.baseJavaType.cast(rawDeployedCi);
		// Cast objects
        final D typedPreviousCi = this.baseJavaType.cast(rawPreviousCi);

        // Log
        AbstractStepsContributor.LOGGER.debug("Add steps to install CI with id [{}]", rawDeployedCi.getId());

        // Do call
        this.update(typedDeployedCi, typedPreviousCi, context, delta);
	}    

	/**
     * Perform type conversion and call uninstall method.
     * 
     * @param context The deployment context used to perform uninstall.
     * @param rawPreviousCi The raw CI type to uninstall.
     * @param delta
     * 			  The delta currently processed.
     */
	protected final void callUninstall(final DeploymentPlanningContext context,
			final Deployed<?, ?> rawPreviousCi, final Delta delta) {

        if (!this.baseJavaType.isInstance(rawPreviousCi) ) {
            // Not correct CI, skip it
            AbstractStepsContributor.LOGGER.error("Unable to process CI [{}]. CI class [{}] doesn't match required (generics) type [{}]", new Object[]{rawPreviousCi.getId(),rawPreviousCi.getClass(), this.baseJavaType});
            throw new InternalPluginException("Found CI that matches CI type criteria (previous deployed CI and container) that doesn't match java template type (generics)");        	
        }		
		
		// Cast objects
        final D typedPreviousCi = this.baseJavaType.cast(rawPreviousCi);

        // Log
        AbstractStepsContributor.LOGGER.debug("Add steps to uninstall CI with id [{}]", rawPreviousCi.getId());

        // Do call
        this.uninstall(typedPreviousCi, context, delta);
	}    

	/**
     * Method to implement CI installation. Called when a CI is newly installed, or updated to be deployed to a new container.
     * 
     * @param ciToInstall
     *            The CI to install.
     * @param context
     *            The deployment planning context.
     * @param delta
     * 			  The delta currently processed.
     */
    protected abstract void install(D ciToInstall, DeploymentPlanningContext context, Delta delta);

    /**
     * Method to implement CI update. Called when a CI will be updated to the same container.
     * 
     * @param newCi
     *            The new CI for the container.
     * @param oldCi
     *            The old/current CI for the container.
     * @param context
     *            The deployment planning context.
     * @param delta
     * 			  The delta currently processed.
     */
    protected abstract void update(D newCi, D oldCi, DeploymentPlanningContext context, Delta delta);

    /**
     * Method to implement CI uninstallation. Called when a CI is removed, or updated to be deployed to a new container.
     * 
     * @param ciToUninstall
     *            The CI to uninstall.
     * @param context
     *            The deployment planning context.
     * @param delta
     * 			  The delta currently processed.
     */
    protected abstract void uninstall(D ciToUninstall, DeploymentPlanningContext context, Delta delta);

}
