from auto_scaling import *
from auto_scaling.group.helpers.vpc_helper import VpcHelper
from ec2.ec2_helper import EC2Helper


class AsgParamsBuilder:
    def __init__(self, deployed):
        self.__params = {}
        self.__deployed = deployed
        self.__ec2_helper = EC2Helper(deployed)
        self.__vpc_helper = VpcHelper(deployed)

    def with_min_size(self, min_size=None):
        self.__params['MinSize'] = min_size if min_size is not None else self.__deployed.minSize
        return self

    def with_max_size(self, max_size=None):
        self.__params['MaxSize'] = max_size if max_size is not None else self.__deployed.maxSize
        return self

    def with_desired_capacity(self, desired_capacity=None):
        self.__params[
            'DesiredCapacity'] = desired_capacity if desired_capacity is not None else self.__deployed.desiredCapacity
        return self

    def with_auto_scale_group_name(self, asg_name=None):
        self.__params['AutoScalingGroupName'] = asg_name if bool(asg_name) else self.__get_asg_name()
        return self

    def with_launch_config_name(self, lc_name=None):
        self.__params['LaunchConfigurationName'] = lc_name if bool(lc_name) else self.__deployed.launchConfigurationName
        return self

    def with_cool_down(self, cool_down=None):
        self.__params['DefaultCooldown'] = cool_down if cool_down is not None else self.__deployed.coolDown
        return self

    def with_new_instances_protected_from_scale_in(self, new_instances_protected_from_scale_in=None):
        self.__params['NewInstancesProtectedFromScaleIn'] = new_instances_protected_from_scale_in if \
            new_instances_protected_from_scale_in is not None else self.__deployed.newInstancesProtectedFromScaleIn
        return self

    def with_termination_policies(self, termination_policies=None):
        self.__params['TerminationPolicies'] = termination_policies if termination_policies is not None else list(
            self.__deployed.terminationPolicies)
        return self

    def with_health_check_type(self, health_check_type=None):
        self.__params['HealthCheckType'] = health_check_type if \
            health_check_type is not None else self.__deployed.healthCheckType
        return self

    def with_health_check_grace_period(self, health_check_grace_period=None):
        self.__params['HealthCheckGracePeriod'] = health_check_grace_period if \
            health_check_grace_period is not None else self.__deployed.healthCheckGracePeriod
        return self

    def with_vpc_zone_identifier(self, subnets=None):
        subnets_to_attach = subnets if subnets is not None else self.__deployed.subnetIds
        self.__params['VPCZoneIdentifier'] = get_formatted_string(self.__vpc_helper.get_subnet_ids(subnets_to_attach))
        return self

    def with_availability_zones(self, zones=None):
        zones_to_attach = zones if zones is not None else self.__deployed.availabilityZones
        self.__params['AvailabilityZones'] = list(zones_to_attach)
        return self

    def with_tags(self, tags=None):
        asg_tags = tags if tags is not None else self.__deployed.asgTags
        self.__params['Tags'] = self.__create_tags(asg_tags)
        return self

    def build(self):
        return self.__params

    def __asg_builder(self):
        return self.with_auto_scale_group_name().with_cool_down() \
            .with_availability_zones().with_vpc_zone_identifier() \
            .with_health_check_type().with_health_check_grace_period() \
            .with_termination_policies().with_new_instances_protected_from_scale_in() \
            .with_min_size().with_max_size().with_desired_capacity()

    def build_update_params(self):
        return self.__asg_builder().build()

    def build_create_params(self):
        return self.__asg_builder().with_launch_config_name().with_tags().build()

    def build_dissociation_params(self, subnets, availability_zones):
        return self.with_auto_scale_group_name().with_vpc_zone_identifier(
            subnets).with_availability_zones(availability_zones).build()

    def __create_tags(self, tags):
        return map(lambda tag: self.__enrich_tag(tag),
                   map(lambda key: {'Key': key, 'Value': tags[key]}, tags.keys()))

    def __get_asg_name(self):
        return self.__deployed.autoScalingGroupName if bool(
            self.__deployed.autoScalingGroupName) else self.__deployed.name

    def __enrich_tag(self, base_tag):
        base_tag['ResourceId'] = self.__get_asg_name()
        base_tag['ResourceType'] = self.__deployed.resourceType
        base_tag['PropagateAtLaunch'] = self.__deployed.propagateAtLaunch
        return base_tag
