from ec2.ec2_helper import EC2Helper
from ec2.vpc.vpc_helper import VPCHelper

class SGHelper(EC2Helper):
    def __init__(self, deployed):
        super(SGHelper, self).__init__(deployed)
        self.vpc_helper = VPCHelper(deployed)

    def create_security_group(self):
        if self.deployed.vpc:
            if self.is_starts_with_name(self.deployed.vpc):
                vpc_id = self.vpc_helper.get_vpc_id_by_name(self.get_property_name(self.deployed.vpc))
            else:
                vpc_id = self.deployed.vpc

            create_sg_response = self.ec2_client.create_security_group(
                GroupName = self.deployed.securityGroupName,
                Description = self.deployed.description,
                VpcId = vpc_id
            )
        else:
            create_sg_response = self.ec2_client.create_security_group(
                GroupName = self.deployed.securityGroupName,
                Description = self.deployed.description
            )

        self.deployed.securityGroupId = create_sg_response['GroupId']
        return create_sg_response

    def delete_security_group(self):
        return self.ec2_client.delete_security_group(
            GroupId=self.deployed.securityGroupId
        )

    def add_inbound_rules(self):
        ipPermissions = []
        for rule in self.deployed.inboundRules:
            ipPermissions.append(self._create_permission(rule))

        self.ec2_client.authorize_security_group_ingress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def add_outbound_rules(self):
        ipPermissions = []
        for rule in self.deployed.outboundRules:
            ipPermissions.append(self._create_permission(rule))

        self.ec2_client.authorize_security_group_egress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def remove_inbound_rules(self):
        ipPermissions = []
        for rule in self.deployed.inboundRules:
            ipPermissions.append(self._create_permission(rule))

        self.ec2_client.revoke_security_group_ingress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def remove_outbound_rules(self):
        ipPermissions = []
        for rule in self.deployed.outboundRules:
            ipPermissions.append(self._create_permission(rule))

        self.ec2_client.revoke_security_group_egress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def remove_ipv4_default_outbound_rules(self):
        ipPermissions = [
            {
                'IpProtocol' : '-1',
                'FromPort' : -1,
                'ToPort' : -1,
                'IpRanges': [
                    {
                        'CidrIp': '0.0.0.0/0'
                    }
                ]
            }
        ]

        self.ec2_client.revoke_security_group_egress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def remove_ipv6_default_outbound_rules(self):
        ipPermissions = [
            {
                'IpProtocol' : '-1',
                'FromPort' : -1,
                'ToPort' : -1,
                'Ipv6Ranges': [
                    {
                        'CidrIpv6': '::/0'
                    }
                ]
            }
        ]

        self.ec2_client.revoke_security_group_egress(
            GroupId = self.deployed.securityGroupId,
            IpPermissions = ipPermissions
        )

    def _create_permission(self, rule):
        permission = {
            'IpProtocol' : rule.protocol,
            'FromPort' : self._get_from_port(rule.portRange),
            'ToPort' : self._get_to_port(rule.portRange)
        }

        target = rule['source'] if hasattr(rule, 'source') else rule['destination']
        if(self.is_ipv4_cidr(target)):
            permission.update({'IpRanges' : [{
                'CidrIp' : target
            }]})
        elif (self.is_ipv6_cidr(target)):
            permission.update({'Ipv6Ranges' : [{
                'CidrIpv6' : target
            }]})
        else:
            if self.is_starts_with_name(target):
                target = self.get_security_group_id(self.get_property_name(target))

            groupPair = {
                'GroupId' : target
            }

            if hasattr(rule, 'vpc') and rule.vpc:
                if self.is_starts_with_name(rule.vpc):
                    vpc_id = self.vpc_helper.get_vpc_id_by_name(self.get_property_name(rule.vpc))
                else:
                    vpc_id = rule.vpc
                groupPair['VpcId'] = vpc_id

            permission.update({'UserIdGroupPairs' : [groupPair]})

        return permission

    def _get_from_port(self, port_range):
        index = port_range.rfind('-')
        if index > 0:
            return int(port_range[:index])
        return int(port_range)

    def _get_to_port(self, port_range):
        index = port_range.rfind('-')
        if index > 0:
            return int(port_range[index + 1:])
        return int(port_range)

    def is_vpc_provided(self):
        return self.deployed.vpc and len(self.deployed.vpc) > 0

    def is_ipv6_cidr(self, source):
        index = source.rfind('/')
        if index >= 0:
            suffix = '::' + source[index:]
            return  source.endswith(suffix)
        return False

    def is_ipv4_cidr(self, source):
        index = source.rfind('/')
        if index >= 0:
            return  source.endswith(source[index:]) and not self.is_ipv6_cidr(source)
        return False

    def is_valid_port_range(self, port_range):
        try:
            self._get_from_port(port_range)
            self._get_to_port(port_range)
            return True
        except:
            return False

    def is_ipv6_default_outbound_rules_exist(self):
        for permission in self._get_outboud_permissions():
            if self._contains(permission['Ipv6Ranges'], lambda range: range['CidrIpv6'] == '::/0') if 'Ipv6Ranges' in permission and permission['IpProtocol'] == '-1' else False:
                return True
        return False

    def is_ipv4_default_outbound_rules_exist(self):
        for permission in self._get_outboud_permissions():
            if self._contains(permission['IpRanges'], lambda range: range['CidrIp'] == '0.0.0.0/0') if 'IpRanges' in permission and permission['IpProtocol'] == '-1' else False:
                return True
        return False

    def _get_outboud_permissions(self):
        security_groups = self.ec2_client.describe_security_groups(GroupIds=[self.deployed.securityGroupId])["SecurityGroups"]
        if len(security_groups) > 0 and "IpPermissionsEgress" in security_groups[0]:
            permissions = security_groups[0]["IpPermissionsEgress"]
        else:
            permissions = []
        return permissions

    def _contains(self, list, filter):
        for x in list:
            if filter(x):
                return True
        return False

    def is_outbound_rules_exist(self):
        return self.deployed.outboundRules and len(self.deployed.outboundRules) > 0

    def is_inbound_rules_exist(self):
        return self.deployed.inboundRules and len(self.deployed.inboundRules) > 0
