'''
Helper functions/classes for alternate authentication methods.
'''

import requests
import re
import base64
from urlparse import urlparse
from bs4 import BeautifulSoup
import xml.etree.ElementTree as ET
import sys
import commons
from boto3.session import Session
from botocore.config import Config
from urlparse import urlparse

class ADFSSAMLAuthHelper(object):
    '''
    Helper for SAMLv2 IdPs, specifically for AD FS 3.0 (Microsoft Active Directory Federation Services).
    This uses form-based authentication, tailored for AD FS 3.0.

    Terms:
        IDP - Identity provider (a SAML term)
        SAML - an XML standard for Single Sign On (SSO).
    '''
    @staticmethod
    def authenticate(container, region, config=None):
        '''
        Authenticates to the IDP and uses the SAMLv2 response to get an access token from AWS STS.

        Args:
            - container - aws.Cloud object containing
                - name
                - verifySSL
                - idpUrl
                - idpVerifySSL
                - idpUsername
                - idpPassword
                - idpRolename - optional. The rolename that should be assumed. Must be found in the SAML response.
            - region - the AWS region to get an STS token for.
        Kwargs:
            - config - botocore.Config for this container, or None.

        Returns:
            - access key id
            - secret access key
            - session token
        '''
        if(not container.idpUrl or not container.idpUsername or not container.idpPassword):
            raise Exception('Required IDP parameters missing.')
        if not region:
            raise Exception('AWS region not specified.')

        # add specific resource to URL
        container.idpUrl += '?loginToRp=urn:amazon:webservices'

        samlResponse = None
        roleARN = None
        principleARN = None
        accessKeyId = None
        secretAccessKey = None

        try:
            samlResponse = ADFSSAMLAuthHelper.__login(container, container.idpUrl,
                container.idpUsername,
                container.idpPassword)
            logger.debug('logged into ' + container.idpUrl + ' user ' + container.idpUsername)
        except Exception as e:
            # helpful detail about the exception
            logger.error('IDP Login failed')
            # reraise, preserving the original stack trace
            raise Exception, Exception(e), sys.exc_info()[2]

        try:
            roleARN, principleARN = ADFSSAMLAuthHelper.__getARNs(samlResponse, container)
            logger.debug('got role/principle ARNs from response')
        except Exception as e:
            logger.error('Failed to parse IDP\'s SAML response for ARNs')
            # reraise, preserving the original stack trace
            raise Exception, Exception(e), sys.exc_info()[2]

        try:
            accessKeyId, secretAccessKey, sessionToken, tokenExpiration = ADFSSAMLAuthHelper.__getSTSToken(container,
                roleARN, principleARN, region, samlResponse, config=config)
            logger.debug('got STS id/token')
        except Exception as e:
            logger.error('Failed to get STS tokens')
            # reraise, preserving the original stack trace
            raise Exception, Exception(e), sys.exc_info()[2]

        logger.debug('Gained STS tokens for ' + container.name + ', expiration: ' + str(tokenExpiration))

        return accessKeyId, secretAccessKey, sessionToken

    @staticmethod
    def __login(container, url, username, password):
        '''
        Args:
            - container - aws.Cloud object containing
                - idpVerifySSL
            - url - IDP URL
            - username
            - password

        Returns SAMLv2 base64-encoded XML assertion.
        '''
        # Programmatically get the SAML assertion
        # Opens the initial IdP url and follows all of the HTTP302 redirects, and
        # gets the resulting login page
        session = requests.Session()

        # form based auth
        formResponse = session.get(url, verify=container.idpVerifySSL)

        # Capture the submitURL, which is the final url after all the 302s
        submitURL = formResponse.url

        def login(response):
            # Parse the response and extract all the necessary values
            # in order to build a dictionary of all of the form values the IdP expects
            formsoup = BeautifulSoup(response.text.decode('utf8'), features='html.parser')
            payload = {}

            for inputtag in formsoup.find_all(re.compile('(INPUT|input)')):
                name = inputtag.get('name','')
                value = inputtag.get('value','')
                logger.debug('found form field ' + name)
                if "username" in name.lower():
                    payload[name] = username
                elif "password" in name.lower():
                    payload[name] = password
                else:
                    # Populate the parameter with existing value (picks up hidden fields in the login form)
                    payload[name] = value

            # check for missing fields. be case insensitive just like the check.
            if 'password' not in map(lambda x: x.lower(), payload.keys()):
                raise Exception('Could not find login form fields.')

            # Some IdPs don't explicitly set a form action, but if one is set we should
            # build the submitURL by combining the scheme and hostname
            # from the entry url with the form action target
            # If the action tag doesn't exist, we just stick with the submitURL above
            for formtag in formsoup.find_all(re.compile('(FORM|form)')):
                action = formtag.get('action')
                logger.debug('found form action ' + action)
                if action:
                    # ADFS 3.0: the action will be the full URL to the server actually, including https://, instead of a /actionURI style action.
                    submitURL = action

            # Performs the submission of the login form with the above post data
            logger.debug('submitURL ' + submitURL)
            response = session.post(submitURL, data=payload, verify=container.idpVerifySSL)
            return response

        def handleResponse(response):
            '''Handle response errors and return the SAML assertion'''
            logger.debug('response.status_code ' + str(response.status_code))

            # check for errors
            if response.status_code != 200:
                raise Exception('Failed to authenticate: Status ' + str(response.status_code) + ' ' + str(response.reason))

            # Decode the response and extract the SAML assertion
            soup2 = BeautifulSoup(response.text.decode('utf8'),"html.parser")
            samlAssertion = ''

            # Look for the SAMLResponse attribute of the input tag (determined by
            # analyzing the debug print lines above)
            for inputtag in soup2.find_all('input'):
                name = inputtag.get('name')
                logger.debug('found form response field ' + str(name))
                if(name == 'SAMLResponse'):
                    samlAssertion = inputtag.get('value')

            return samlAssertion

        response = login(formResponse)
        assertion = handleResponse(response)

        # sometimes it thinks we're already logged in and asks us to pick a site...
        if (assertion == ''):
            payload = {}
            # collect relying party value and proper input fields
            soup = BeautifulSoup(response.text.decode('utf8'),"html.parser")
            for tag in soup.find_all(re.compile('(input|INPUT|option|OPTION)')):
                name = tag.get('name')
                logger.debug('2 found form field ' + str(name))
                logger.debug('2 found form field contents ' + str(tag.contents))
                # site option tag
                if(len(tag.contents) > 0 and str(tag.contents[0]) == 'Amazon Web Services'):
                    payload['RelyingParty'] = tag.get('value')
                elif(name == 'SignInGo'):
                    payload[name] = tag.get('value')
                elif(name == 'SignInOtherSite'):
                    payload[name] = tag.get('value')
                # any other field that we dont need.
                elif(name is not None
                    and not name == 'SignInIdpSite'
                    and not name == 'SignInSubmit'
                    and not name == 'SingleSignOut'
                    and not name == 'LocalSignOut'
                    and not name == 'SignOut'):
                    payload[name] = tag.get('value')

            # get appropriate form action
            for formtag in soup.find_all(re.compile('(FORM|form)')):
                action = formtag.get('action')
                logger.debug('2 found form action ' + action)
                if action:
                    parsedurl = urlparse(submitURL)
                    submitURL = parsedurl.scheme + "://" + parsedurl.netloc + action

            # Performs the submission of the login form with the above post data
            logger.debug('2 submitURL ' + submitURL)
            response = session.post(submitURL, data=payload, verify=container.idpVerifySSL)

            assertion = handleResponse(response)

        if (assertion == ''):
            # try once more
            logger.warn('No SAML response, attempting login again')
            print 'No SAML response, attempting login again'

            response = login(response)
            assertion = handleResponse(response)
            if (assertion == ''):
                raise Exception('Response did not contain a valid SAML assertion')

        return assertion

    @staticmethod
    def __getARNs(samlResponse, container):
        '''
        Parse SAML response for ARNs.

        Args:
            - samlResponse - base64 encoded XML string containing SAML assertion containing attributes used by AWS.
            - container - an object containing
                - idpRolename - optional. The rolename that should be assumed. Must be found in the SAML response.
        Returns:
            - roleARN - the role to assume
            - principleARN - the ARN of the SAML IDP (identity provider).
        '''
        # grab AWS roles from the XML SAML assertion
        awsRoles = []
        root = ET.fromstring(base64.b64decode(samlResponse))
        for saml2attribute in root.iter('{urn:oasis:names:tc:SAML:2.0:assertion}Attribute'):
            if (saml2attribute.get('Name') == 'https://aws.amazon.com/SAML/Attributes/Role'):
                for saml2attributevalue in saml2attribute.iter('{urn:oasis:names:tc:SAML:2.0:assertion}AttributeValue'):
                    awsRoles.append(saml2attributevalue.text)

        if len(awsRoles) == 0:
            raise Exception('No ARNs found in SAML assertion')

        # fix arn order
        # Note the format of the attribute value should be role_arn,principal_arn but lots of blogs list it as principal_arn,role_arn
        # so let's reverse them if needed
        for awsRole in awsRoles:
            chunks = awsRole.split(',')
            if'saml-provider' in chunks[0]:
                newAwsRole = chunks[1] + ',' + chunks[0]
                index = awsRoles.index(awsRole)
                awsRoles.insert(index, newAwsRole)
                awsRoles.remove(awsRole)

        selectedRole = None
        if container.idpRolename:
            # if specified, search roles for that ROlename (end of roleARN is role name)
            for awsRole in awsRoles:
                role = awsRole.split(',')[0]
                if role.endswith(container.idpRolename):
                    selectedRole = awsRole

            if not selectedRole:
                raise Exception('Requested rolename not found in assertion.')

            logger.debug('found desired role')
        else:
            selectedRole = awsRoles[0]
            logger.debug('chose first role ARN')

        roles = selectedRole.split(',')
        roleARN = roles[0]
        principalARN = roles[1]

        return roleARN, principalARN

    @staticmethod
    def __getSTSToken(container, roleARN, principleARN, region, samlResponse, config=None):
        '''
        Use the provided ARNs with AWS STS service to get temporary security tokens.

        Args:
            - container - aws.Cloud object containing
                - verifySSL
            - roleARN - the role to assume
            - principleARN - the ARN of the SAML IDP (identity provider).
            - region - the AWS region to get an STS token for
            - samlResponse - the base64 encoded XML string containing the IDP's SAML assertion.
        Kwargs:
            - config - botocore.Config for this container, or None.

        Returns:
            - accessKeyId - ID of the key. Used to authenticate to AWS.
            - secretAccessKey - The key. Used to authenticate to AWS.
            - sessionToken
            - tokenExpiration - Number describing the expiration time of the token.
        '''
        # Use the assertion to get an AWS STS token using Assume Role with SAML
        botocore_session = commons.get_botocore_session()
        session_keywords = {
            'botocore_session': botocore_session
        }
        botosession = Session(**session_keywords)

        client = None
        if not container.verifySSL:
            client = botosession.client('sts', region_name=region, verify=False, config=config)
        else:
            client = botosession.client('sts', region_name=region, config=config)

        token = client.assume_role_with_saml(
            RoleArn=roleARN,
            PrincipalArn=principleARN,
            SAMLAssertion=samlResponse
        )

        accessKeyId = token["Credentials"]["AccessKeyId"]
        secretAccessKey = token["Credentials"]["SecretAccessKey"]
        sessionToken = token["Credentials"]["SessionToken"]
        tokenExpiration = token["Credentials"]["Expiration"]

        return accessKeyId, secretAccessKey, sessionToken, tokenExpiration
