import os
import tempfile
from java.nio.file import Files, Paths
import urllib
import commons
import com.xebialabs.deployit.plugin.aws.TrustExtractHelper as TrustExtractHelper
from auth_helper import ADFSSAMLAuthHelper
from boto3.session import Session
from botocore.config import Config


class AWSHelper(object):

    def __init__(self, deployed):
        self.deployed = deployed
        AWSHelper.__validate(deployed.container)
        AWSHelper.set_ca_bundle_path()
        self.session = AWSHelper.create_session(deployed.container,
                                                deployed.region if hasattr(deployed, 'region') else container.region)

    @staticmethod
    def create_session(container, region_name=None):
        '''
        Region param is needed for IDP auth to AWS STS token service.
        '''
        # SSO login
        if container.sessionToken:
            session = AWSHelper.__get_session(use_credentials=True, access_key=container.accesskey,
                                              access_secret=container.accessSecret,
                                              aws_session_token=container.sessionToken)
        elif container.idpUrl:
            session = AWSHelper.auth_with_idp(container, region)
        else:
            # Normal login
            session = AWSHelper.__get_session(container.useCredentials, container.accesskey, container.accessSecret)

        if container.roleName:
            response = AWSHelper.__get_assume_role_response(session, container, region_name=region_name)
            session = AWSHelper.__get_session(use_credentials=True,
                                              access_key=response['Credentials']['AccessKeyId'],
                                              access_secret=response['Credentials']['SecretAccessKey'],
                                              aws_session_token=response['Credentials']['SessionToken'])
        return session

    @staticmethod
    def auth_with_idp(container, region):
        # get token
        # since we are connecting to AWS we need to set CA bundle path.
        AWSHelper.set_ca_bundle_path()
        config = AWSHelper.create_config(container)
        accessKey, accessSecret, sessionToken = ADFSSAMLAuthHelper.authenticate(container, region, config=config)
        session = AWSHelper.__get_session(container.useCredentials, accessKey, accessSecret,
                                          aws_session_token=sessionToken)
        return session

    @staticmethod
    def __get_assume_role_response(session, container, region_name=None):
        sts_client = AWSHelper.__get_sts_client(session, container, region_name=region_name)
        assume_keywords = {
            'RoleArn': AWSHelper.__get_role_arn(container.roleName, container.accountId),
            'RoleSessionName': container.name,
            'DurationSeconds': container.durationSeconds if container.durationSeconds else 3600
        }
        if container.externalId:
            assume_keywords['ExternalId'] = container.externalId

        response = sts_client.assume_role(**assume_keywords)
        assert response['Credentials'], 'Assume role failed, response: {0}'.format(response)
        return response

    @staticmethod
    def __get_role_arn(role_name, account_id):
        return 'arn:aws:iam::{0}:role/{1}'.format(account_id, role_name)

    @staticmethod
    def __get_sts_client(session, container, region_name=None):
        config = AWSHelper.create_config(container)
        if region_name == '':
            region_name = None
        if not container.verifySSL:
            return session.client('sts', region_name=region_name, verify=False, config=config)
        return session.client('sts', region_name=region_name, config=config)

    @staticmethod
    def __validate(container):
        if container.useCredentials:
            if container.idpUrl:
                assert container.idpUsername, "IDP Username should not be empty"
                assert container.idpPassword, "IDP Password should not be empty"
            else:
                assert container.accesskey, "Access key should not be empty."
                assert container.accessSecret, "Access secret should not be empty."
        if container.roleName or container.accountId or container.externalId:
            assert container.accountId, "Assume Role's account id should not be empty when trying to assume role."
            assert container.roleName, "Assume Role's role name should not be empty when trying to assume role."

        if (container.proxyHost or container.proxyPort or container.proxyProtocol) and \
                not (container.proxyHost and container.proxyProtocol and container.proxyPort):
            raise Exception("Either all or none of proxy host, port and protocol should be provided.")

    @staticmethod
    def __get_session(use_credentials, access_key, access_secret, aws_session_token=None):
        botocore_session = commons.get_botocore_session()
        session_keywords = {
            'botocore_session': botocore_session
        }
        if use_credentials:
            session_keywords['aws_access_key_id'] = access_key
            session_keywords['aws_secret_access_key'] = access_secret

        if aws_session_token:
            session_keywords['aws_session_token'] = aws_session_token

        return Session(**session_keywords)

    def get_aws_client(self, region, resource_name='ec2'):
        config = AWSHelper.create_config(self.deployed.container)
        if not self.deployed.container.verifySSL:
            return self.session.client(resource_name, region_name=region, verify=False, config=config)
        return self.session.client(resource_name, region_name=region, config=config)

    @staticmethod
    def create_config(container):
        config = None
        if container.proxyHost and container.proxyProtocol and container.proxyPort:
            proxy = "{}:{}".format(container.proxyHost, container.proxyPort)
            if container.proxyUser and container.proxyPassword:
                proxy = "{}:{}@{}".format(urllib.quote(container.proxyUser), urllib.quote(container.proxyPassword),
                                          proxy)
            if container.proxyProtocol is "https":
                proxies = {'https': proxy}
            else:
                proxies = {'http': proxy}
            config = Config(proxies=proxies)

        return config

    @staticmethod
    def create_tmp_pem(pemStr):
        if pemStr:
            tmp_file, tmp_abs_path = tempfile.mkstemp()
            tmp_file.close()
            Files.write(Paths.get(tmp_abs_path), bytes(pemStr))
            return tmp_abs_path
        else:
            return None

    @staticmethod
    def set_ca_bundle_path():
        os.environ['REQUESTS_CA_BUNDLE'] = TrustExtractHelper.getCACertPath()

    @staticmethod
    def is_starts_with_name(property_value):
        return property_value.lower().startswith('name:') if property_value else False

    @staticmethod
    def get_property_name(property_name):
        return property_name[5:]

    @staticmethod
    def is_success(response):
        return 299 >= response['ResponseMetadata']['HTTPStatusCode'] >= 200

    @staticmethod
    def remove_none_keys(dict):
        return {k: v for k, v in dict.iteritems() if v is not None}

    @staticmethod
    def remove_empty_and_none_values(dict):
        return {k: v for k, v in dict.iteritems() if (bool(v) if isinstance(v, (list, set)) else v is not None)}

    @staticmethod
    def get_current_retry_count(context, counter_name_suffix):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        current_retry_count = context.getAttribute(counter_name_suffix)
        current_retry_count = 1 if not current_retry_count else current_retry_count
        return current_retry_count

    @staticmethod
    def increment_retry_counter(context, counter_name_suffix):
        current_retry_count = AWSHelper.get_current_retry_count(context, counter_name_suffix)
        current_retry_count = current_retry_count + 1
        AWSHelper.set_current_retry_count(context, counter_name_suffix, current_retry_count)

    @staticmethod
    def set_current_retry_count(context, counter_name_suffix, current_retry_count):
        counter_name_suffix = "current_retry_{0}".format(counter_name_suffix)
        context.setAttribute(counter_name_suffix, current_retry_count)

    def retry_or_fail(self, context, subject, max_retry_count, fail_message, wait_message):
        retry_count = self.get_current_retry_count(context, "{0}_stopped".format(subject))
        if retry_count > max_retry_count:
            raise RuntimeError("Reached maximum limit of {0} retries. {1}"
                               .format(max_retry_count, fail_message))
        else:
            print("{0} Done with retry {1}".format(wait_message, retry_count))
            self.increment_retry_counter(context, "{0}_stopped".format(subject))
            return "RETRY"

    def get_deployable_name(self, deployed_id=None):
        deployed_id = self.deployed.id if not deployed_id else deployed_id
        return deployed_id[deployed_id.rfind('/') + 1:]

    @staticmethod
    def check_connection(container, region):
        # this assertion only applies to check connection operation.
        AWSHelper.__validate(container)
        AWSHelper.set_ca_bundle_path()
        if container.idpUrl:
            if not region:
                raise Exception("Check Connection Region should not be empty when IDP URL present on container.")

            session = AWSHelper.auth_with_idp(container, region)
        else:
            if container.sessionToken:
                session = AWSHelper.__get_session(container.useCredentials, container.accesskey, container.accessSecret, container.sessionToken)
            else:
                session = AWSHelper.__get_session(container.useCredentials, container.accesskey, container.accessSecret)

        print('Verifying connection.')
        client = AWSHelper.__get_sts_client(session, container)
        client.get_caller_identity()

        if container.roleName:
            print(
                'Trying to assume role {0}.'.format(AWSHelper.__get_role_arn(container.roleName, container.accountId)))
            AWSHelper.__get_assume_role_response(session, container, region_name=region)

        return True

    @staticmethod
    def submit_token(container, mfaToken, tokenTimeOutDuration, mfaSerialNumber):
        assert mfaToken, "MFA token code cannot be empty"
        assert tokenTimeOutDuration, "Timeout Duration cannot be empty"
        assert mfaSerialNumber, "MFA Serial number cannot be empty"
        session = AWSHelper.__get_session(True, container.accesskey, container.accessSecret)
        sts_client = session.client('sts', verify=False)
        temp_credentials = sts_client.get_session_token(
            DurationSeconds=int(tokenTimeOutDuration),
            SerialNumber=mfaSerialNumber,
            TokenCode=mfaToken
        )
        container.accesskey = temp_credentials['Credentials']['AccessKeyId']
        container.accessSecret = temp_credentials['Credentials']['SecretAccessKey']
        container.sessionToken = temp_credentials['Credentials']['SessionToken']
        container.sessionExpiry = (temp_credentials['Credentials']['Expiration']).strftime("%m/%d/%Y, %H:%M:%S")
        print("Successfully submitted the token")
