from ec2.ec2_helper import EC2Helper


class VPCHelper(EC2Helper):
    def __init__(self, deployed):
        super(VPCHelper, self).__init__(deployed)

    def create_vpc(self):
        response = self.ec2_client.create_vpc(CidrBlock=self.deployed.cidrBlock,
                                              InstanceTenancy=self.deployed.instanceTenancy,
                                              AmazonProvidedIpv6CidrBlock=self.deployed.amazonProvidedIpv6CidrBlock)
        return response

    def set_vpc_name(self, vpc_id, vpc_name):
        self.ec2_client.create_tags(DryRun=False, Resources=[vpc_id],
                                    Tags=[{'Key': 'Name', 'Value': vpc_name}])

    def is_vpc_created(self):
        response = self.ec2_client.describe_vpcs(DryRun=False,
                                                 VpcIds=[self.deployed.vpcId])
        if response['ResponseMetadata']['HTTPStatusCode'] != 200 or len(response['Vpcs']) == 0:
            raise RuntimeError("No VPC found with id %s" % self.deployed.vpcId)
        elif len(response['Vpcs']) > 1:
            raise RuntimeError("Multiple Volume found with id %s" % self.deployed.volumeId)
        else:
            return response['Vpcs'][0]['State'] == 'available'

    def destroy_vpc(self):
        self.ec2_client.delete_vpc(VpcId=self.deployed.vpcId)
        print "VPC %s is deleted successfully." % self.deployed.vpcId

    def enable_vpc_classic_link(self):
        self.ec2_client.enable_vpc_classic_link(VpcId=self.deployed.vpcId)
        print "Classic link is enabled for VPC %s " % self.deployed.vpcId

    def disable_vpc_classic_link(self):
        self.ec2_client.disable_vpc_classic_link(VpcId=self.deployed.vpcId)
        print "Classic link is disabled for VPC %s " % self.deployed.vpcId

    def enable_vpc_classic_link_dns_support(self):
        self.ec2_client.enable_vpc_classic_link_dns_support(VpcId=self.deployed.vpcId)
        print "Classic link DNS support is enabled for VPC %s " % self.deployed.vpcId

    def disable_vpc_classic_link_dns_support(self):
        self.ec2_client.disable_vpc_classic_link_dns_support(VpcId=self.deployed.vpcId)
        print "Classic link DNS support is disabled for VPC %s " % self.deployed.vpcId

    def modify_dns_support(self):
        self.ec2_client.modify_vpc_attribute(VpcId=self.deployed.vpcId,
                                             EnableDnsSupport={'Value': self.deployed.dnsSupport})
        print "DNS support is modified to %s for VPC %s " % (self.deployed.dnsSupport, self.deployed.vpcId)

    def modify_dns_host_names(self):
        self.ec2_client.modify_vpc_attribute(VpcId=self.deployed.vpcId,
                                             EnableDnsHostnames={'Value': self.deployed.dnsHostnames})
        print "DNS host name support is modified to %s for VPC %s " % (self.deployed.dnsHostnames, self.deployed.vpcId)

    def modify_amazon_vpc_cidr_block(self):
        self.ec2_client.associate_vpc_cidr_block(VpcId=self.deployed.vpcId,
                                                 AmazonProvidedIpv6CidrBlock=self.deployed.amazonProvidedIpv6CidrBlock)
        print "Amazon VPC cidr_block is modified to %s for VPC %s " % (
        self.deployed.amazonProvidedIpv6CidrBlock, self.deployed.vpcId)

    def get_vpc_id_by_name(self, vpc_name):
        vpc = self.get_vpc_by_name(vpc_name)
        return vpc['VpcId']

    def get_vpc_by_name(self, vpc_name):
        response = self.ec2_client.describe_vpcs(Filters=[
            {
                'Name': 'tag:Name',
                'Values': [vpc_name]
            },
        ])
        if not self.is_success(response) or len(response['Vpcs']) == 0:
            raise Exception("VPC with name %s not found in AWS." % vpc_name)
        elif len(response['Vpcs']) > 1:
            raise Exception("More than one VPC found with name %s." % vpc_name)
        else:
            return response['Vpcs'][0]

    def get_vpc_by_id(self, vpc_id):
        response = self.ec2_client.describe_vpcs(VpcIds=[vpc_id])
        if not self.is_success(response) or len(response['Vpcs']) == 0:
            raise Exception("VPC with id %s not found in AWS." % vpc_id)
        else:
            return response['Vpcs'][0]

    def accept_vpc_peer_connection(self, vpc_peer_connection_id):
        self.ec2_client.accept_vpc_peering_connection(VpcPeeringConnectionId=vpc_peer_connection_id)

    def create_vpc_peer_connection(self, peer_vpc):
        if self.is_starts_with_name(peer_vpc):
            peer_vpc = self.get_vpc_id_by_name(self.get_property_name(peer_vpc))

        response = self.ec2_client.create_vpc_peering_connection(
            PeerVpcId=peer_vpc,
            VpcId=self.deployed.vpcId
        )
        return response["VpcPeeringConnection"]["VpcPeeringConnectionId"]

    def delete_vpc_peer_connection(self, peer_vpc):
        if self.is_starts_with_name(peer_vpc):
            peer_vpc = self.get_vpc_id_by_name(self.get_property_name(peer_vpc))

        response = self.ec2_client.describe_vpc_peering_connections(Filters=[
            {
                'Name': 'requester-vpc-info.vpc-id',
                'Values': [self.deployed.vpcId]
            },
            {
                'Name': 'accepter-vpc-info.vpc-id',
                'Values': [peer_vpc]
            }
        ])

        vpc_peer_connection_id = response["VpcPeeringConnections"][0]["VpcPeeringConnectionId"]
        self.ec2_client.delete_vpc_peering_connection(VpcPeeringConnectionId=vpc_peer_connection_id)
