import re

from elb_utils import ElbUtils


class ElbValidator:
    def __init__(self, deployed, previous_deployed):
        self.deployed = deployed
        self.load_balancer_name_regex = re.compile("[0-9a-zA-Z]+[-]*[a-zA-Z0-9]+")
        self.previous_deployed = previous_deployed
        self.elb_util = ElbUtils(deployed)

    def should_not_modify_load_balancer_name(self):
        if self.deployed.loadBalancerName != self.previous_deployed.loadBalancerName:
            raise RuntimeError("Load balancer name cannot be changed during modification")
        return True

    def should_not_modify_scheme(self):
        if self.deployed.scheme != self.previous_deployed.scheme:
            raise RuntimeError("Scheme cannot be changed during modification")
        return True

    def should_not_use_subnets_and_availabiliy_zones_simultaneously(self):
        if bool(self.deployed.subnetIds) and bool(self.deployed.availabilityZones):
            raise RuntimeError("You cannot enable availability zones for ELBs in VPC.")
        return True

    def should_have_minimum_subnets(self, min_value):
        if not bool(self.deployed.availabilityZones) and len(self.deployed.subnetIds) < min_value:
            raise RuntimeError("Cannot have less than %s subnets" % min_value)

    def should_have_minimum_listeners(self, min_value):
        if len(self.deployed.listeners) < min_value:
            raise RuntimeError("Cannot have less than %s listeners" % min_value)

    def should_not_have_multiple_listeners_with_same_load_balancer_port(self):
        ports = map(lambda listener: listener.loadBalancerPort, self.deployed.listeners)
        if not len(set(ports)) == len(ports):
            raise RuntimeError("Multiple listeners cannot have the same load balancer port")
        return True

    def should_have_a_valid_load_balancer_name(self):
        is_match = len(self.load_balancer_name_regex.match(self.deployed.loadBalancerName).group()) == len(
            self.deployed.loadBalancerName)
        if not is_match:
            raise RuntimeError("Invalid ELB name %s" % self.deployed.loadBalancerName)
        return is_match

    def should_configure_elb(self):
        return self.should_add_attributes() or \
               self.should_attach_ec2_instances() or \
               self.should_create_app_stickiness_policy(self.deployed.listeners) or \
               self.should_create_lb_stickiness_policy(self.deployed.listeners)

    def should_dissociate_resources(self):
        return self.should_detach_subnets() or \
               self.should_detach_availability_zones() or \
               self.should_detach_ec2_instances() or \
               self.should_detach_security_groups()

    def should_associate_resources(self):
        return self.should_attach_subnets() or \
               self.should_attach_availability_zones() or \
               self.should_attach_ec2_instances() or \
               self.should_modify_attributes() or \
               self.should_attach_security_groups()

    def should_add_attributes(self):
        return bool(self.deployed.enableCrossZoneLoadBalancing) or \
               bool(self.deployed.enableConnectionDraining) or \
               self.should_enable_access_log()

    def should_enable_access_log(self):
        if self.deployed.enableAccessLog and bool(self.deployed.accessLogS3BucketName.strip()):
            raise RuntimeError("Please provide s3 bucket name if access log is enabled")

        return self.deployed.enableAccessLog and bool(self.deployed.accessLogS3BucketName.strip())

    def should_create_app_stickiness_policy(self, listeners):
        return bool(filter
                    (lambda listener: self.is_valid_app_cookie_policy(listener),
                     map(lambda listener: listener, listeners)
                     ))

    def should_create_lb_stickiness_policy(self, listeners):
        return bool(filter(
            lambda listener: bool(listener.enableLbCookieStickiness),
            map(lambda listener: listener, listeners))
        )

    def should_attach_subnets(self):
        return self.is_elb_in_vpc() and bool(
            self.elb_util.get_delta(set1=self.deployed.subnetIds, set2=self.previous_deployed.subnetIds))

    def should_detach_subnets(self):
        return self.is_elb_in_vpc() and bool(
            self.elb_util.get_delta(set1=self.previous_deployed.subnetIds, set2=self.deployed.subnetIds))

    def should_modify_security_groups(self):
        return self.is_elb_in_vpc() and self.elb_util.is_set_modified(prev=self.previous_deployed.securityGroups,
                                                                      cur=self.deployed.securityGroups)

    def should_attach_security_groups(self):
        security_groups_to_be_attached = self.elb_util.get_delta(set1=self.deployed.securityGroups,
                                                                 set2=self.previous_deployed.securityGroups)
        return bool(security_groups_to_be_attached)

    def should_detach_security_groups(self):
        security_groups_to_be_detached = self.elb_util.get_delta(set1=self.previous_deployed.securityGroups,
                                                                 set2=self.deployed.securityGroups)
        return bool(security_groups_to_be_detached)

    def should_detach_availability_zones(self):
        return bool(self.elb_util.get_delta(set1=self.previous_deployed.availabilityZones,
                                            set2=self.deployed.availabilityZones))

    def should_attach_availability_zones(self):
        return bool(self.elb_util.get_delta(set1=self.deployed.availabilityZones,
                                            set2=self.previous_deployed.availabilityZones))

    def should_modify_attributes(self):
        return self.is_cross_zone_load_balancing_modified() or \
               self.is_connection_draining_modified() or \
               self.is_connection_setting_idle_timeout_modified() or \
               self.is_access_log_modified()

    def should_detach_ec2_instances(self):
        return bool(
            self.elb_util.get_delta(set1=self.previous_deployed.ec2InstanceIds, set2=self.deployed.ec2InstanceIds))

    def should_attach_ec2_instances(self):
        return bool(self.elb_util.get_delta(set1=self.deployed.ec2InstanceIds,
                                            set2=self.previous_deployed.ec2InstanceIds)) if bool(
            self.previous_deployed) else bool(self.deployed.ec2InstanceIds)

    def should_have_at_least_one_subnet_unmodified(self):
        if self.elb_util.has_no_common_elements(set1=self.deployed.subnetIds, set2=self.previous_deployed.subnetIds) and \
            self.is_elb_in_vpc():
            raise RuntimeError(
                "You are not able to change all the subnets associated with a load balancer at the same time")

    def should_have_at_least_one_security_group_unmodified(self):
        if self.is_elb_in_vpc() and \
            bool(self.previous_deployed.securityGroups) and \
            self.elb_util.has_no_common_elements(set1=self.deployed.securityGroups,
                                                 set2=self.previous_deployed.securityGroups):
            raise RuntimeError(
                'You are not able to change all the security groups associated with a load balancer at the same time')

    def should_not_have_listeners_with_multiple_policies_defined(self):
        is_any_listener_with_multiple_policies = reduce(lambda prev, next: prev or next, [
            (bool(listener.enableLbCookieStickiness) and bool(listener.enableAppCookieStickiness))
            for listener in self.deployed.listeners])

        if is_any_listener_with_multiple_policies:
            raise RuntimeError("Listener cannot have both app and lb cookie policy set at the same time")

    def is_cross_zone_load_balancing_modified(self):
        return self.elb_util.is_ci_property_modified(prev=self.previous_deployed.enableCrossZoneLoadBalancing,
                                                     cur=self.deployed.enableCrossZoneLoadBalancing)

    def is_connection_draining_modified(self):
        return self.elb_util.is_ci_property_modified(prev=self.previous_deployed.enableConnectionDraining,
                                                     cur=self.deployed.enableConnectionDraining) or \
               self.elb_util.is_ci_property_modified(prev=self.previous_deployed.connectionDrainingTimeout,
                                                     cur=self.deployed.connectionDrainingTimeout)

    def is_connection_setting_idle_timeout_modified(self):
        return self.elb_util.is_ci_property_modified(prev=self.previous_deployed.connectionSettingIdleTimeout,
                                                     cur=self.deployed.connectionSettingIdleTimeout)

    def is_access_log_modified(self):
        return self.elb_util.is_ci_property_modified(prev=self.previous_deployed.enableAccessLog,
                                                     cur=self.deployed.enableAccessLog) or \
               self.elb_util.is_ci_property_modified(prev=self.previous_deployed.accessLogS3BucketName,
                                                     cur=self.deployed.accessLogS3BucketName) or \
               self.elb_util.is_ci_property_modified(prev=self.previous_deployed.accessLogS3BucketPrefix,
                                                     cur=self.deployed.accessLogS3BucketPrefix) or \
               self.elb_util.is_ci_property_modified(prev=self.previous_deployed.accessLogEmitInterval,
                                                     cur=self.deployed.accessLogEmitInterval)

    def is_valid_app_cookie_policy(self, listener):
        return bool(listener.enableAppCookieStickiness) and bool(listener.appCookieName) and bool(
            listener.appCookieName.strip())

    def is_elb_in_vpc(self):
        return bool(self.deployed.subnetIds) or bool(self.previous_deployed.subnetIds) \
            if bool(self.previous_deployed) else bool(self.deployed.subnetIds)
