#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

from overtherepy import OverthereHostSession
import sys


class CfRunner:
    def __init__(self, cfclient):
        self.cfclient = cfclient
        self._preview = False
        self.options = {}

    def get_cf_command(self):
        cf = '{0}/cf'.format(self.cfclient.cfHome)
        if self.cfclient.debug:
            cf = cf + ' -v'
        return cf

    def command_line(self, session, deployed):
        raise Exception("Not Implemented")

    def preview(self, deployed):
        session = OverthereHostSession(self.cfclient.host, stream_command_output=False)
        try:
            self._preview = True
            command_line = self.command_line(session, deployed)
            print(command_line)
        finally:
            session.close_conn()

    def execute(self, deployed):
        session = OverthereHostSession(self.cfclient.host, stream_command_output=False)
        try:
            command_line = self.command_line(session, deployed)
            print(command_line)
            uploaded_runner = session.upload_text_content_to_work_dir(command_line, 'xldeploy_cf.sh', executable=True)
            print(uploaded_runner.path)
            response = session.execute(command_line, check_success=False)
            print "\n".join(response.stdout)
            print "\n".join(response.stderr)
            rc = response.rc
            if response.rc > 0:
                sys.exit(rc)
        finally:
            session.close_conn()

    def generate_variable(self, deployed):
        vars = ["--var {0}={1}".format(k, v) for k, v in deployed.inputVariables.items()]
        if self._preview:
            vars.extend(["--var {0}=********".format(k) for k, v in deployed.secretInputVariables.items()])
        else:
            vars.extend(["--var {0}={1}".format(k, v) for k, v in deployed.secretInputVariables.items()])
        return " ".join(vars)


def _convert_dict_to_cmd_opts(opts):
    if not opts:
        return ""
    cmd_opts = ""
    for k, v in opts.items():
        cmd_opts += v
    return cmd_opts


class CfPush(CfRunner):
    def command_line(self, session, deployed):
        return "{0} push {1} {2}".format(self.get_cf_command(), deployed.appName, self._populate_options(session, deployed))

    def _populate_options(self, session, deployed):
        vars_file_path = ""
        for vf in deployed.varFiles:
            uploaded_var_file = session.upload_file_to_work_dir(vf.getFile())
            vars_file_path = vars_file_path + " --vars-file " + uploaded_var_file.getPath()
        app_zip_path = ""
        for app_zip in deployed.appFile:
            uploaded_zip_file = session.upload_file_to_work_dir(app_zip.getFile())
            app_zip_path = app_zip_path + " -p " + uploaded_zip_file.getPath()
        print(app_zip_path)
        manifest_file_path = ""
        for man_file in deployed.manifestFile:
            uploaded_man_file = session.upload_file_to_work_dir(man_file.getFile())
            manifest_file_path = manifest_file_path + " -f " + uploaded_man_file.getPath()
        if deployed.buildPack:
            self.options['buildpack'] = " -b {} ".format(deployed.buildPack)
        if deployed.command:
            self.options['command'] = " -c {} ".format(deployed.command)
        if deployed.dockerImage:
            self.options['dockerImage'] = " --docker-image {} ".format(deployed.dockerImage)
        if deployed.dockerUsername:
            self.options['dockerUsername'] = " --docker-username {} ".format(deployed.dockerUsername)
        if not deployed.ignoreManifest and len(deployed.manifestFile) > 0:
            self.options['manifest'] = manifest_file_path
        if deployed.ignoreManifest:
            self.options['ignoreManifest'] = " --no-manifest "
        if deployed.noStart:
            self.options['noStart'] = " --no-start "
        if deployed.numInstances:
            self.options['numInstances'] = " -i {} ".format(deployed.numInstances)
        if deployed.disk:
            self.options['DISK'] = " -k {} ".format(deployed.disk)
        if deployed.memory:
            self.options['memory'] = " -m {} ".format(deployed.memory)
        if len(deployed.appFile) > 0:
            self.options['path'] = app_zip_path
        if deployed.stack:
            self.options['stack'] = " -s {} ".format(deployed.stack)
        if deployed.noWait and self.cfclient.cliVersion == "v7":
            self.options['noWait'] = " --no-wait "
        if deployed.endpoint and self.cfclient.cliVersion == "v7":
            self.options['endpoint'] = " --endpoint {} ".format(deployed.endpoint)
        if deployed.healthTimeout:
            self.options['healthTimeout'] = " -t {} ".format(deployed.healthTimeout)
        if deployed.healthCheckType:
            self.options['healthCheckType'] = " -u {} ".format(deployed.healthCheckType)
        if deployed.route:
            if deployed.route == "hostname":
                self.options['route'] = " --hostname {} ".format(deployed.hostName)
            self.options['route'] = " --{} ".format(deployed.route)
        if deployed.domain:
            self.options['domain'] = " -d {} ".format(deployed.domain)
        if deployed.routePath:
            self.options['routePath'] = " --route-path {} ".format(deployed.routePath)
        if deployed.strategy and self.cfclient.cliVersion == "v7":
            self.options['strategy'] = " --strategy {} ".format(deployed.strategy)
        if len(deployed.varFiles) > 0:
            self.options['varFiles'] = vars_file_path
        return _convert_dict_to_cmd_opts(self.options) + " " + self.generate_variable(deployed)
