#
# Copyright (c) 2021. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

from java.io import File
import yaml

class Generator:
    def __init__(self, container):
        self.container = container
        self._preview = False
        self.options = {}

    def get_cf_command(self):
        cf = '{0}/cf'.format(self.container.cfHome)
        return cf

    def preview(self, deployed):
        self._preview = True
        command_line = self.command_line(deployed)
        print(command_line)

    def execute(self, deployed):
        command_line = self.command_line(deployed)
        print(command_line)
        return command_line

    def generate_variable(self, deployed):
        vars = ["--var {0}={1}".format(k, v) for k, v in deployed.inputVariables.items()]
        if self._preview:
            vars.extend(["--var {0}=********".format(k) for k, v in deployed.secretInputVariables.items()])
        else:
            vars.extend(["--var {0}={1}".format(k, v) for k, v in deployed.secretInputVariables.items()])
        return " ".join(vars)

    def get_manifest_path(self, deployed):
        return "{}{}{}".format(deployed.file.path,File.separator,deployed.manifestFileName)

    def get_applications_names(self, deployed):
        yaml_path = self.get_manifest_path(deployed)
        with open(yaml_path, 'r') as stream:
            app_names = []
            for app in yaml.load(stream)['applications']:
                app_names.append(app['name'])
            return app_names

    def deleteManifest(self, deployed):
        app_names = self.get_applications_names(deployed)
        commands = []
        for app in app_names:
            command_line = "{0} delete {1} -f -r".format(self.get_cf_command(), app)
            print(command_line)
            commands.append(command_line)
        return commands

    def command_line(self, deployed):
        try:
            if deployed.manifestFileName:
                return "{0} push -f {1}/{2}".format(self.get_cf_command(), deployed.file.path, deployed.manifestFileName)
        except AttributeError:
            return "{0} push {1} {2}".format(self.get_cf_command(), deployed.appName, self._populate_options(deployed))

    def _populate_options(self, deployed):
        app_path = " -p " + deployed.file.path
        self.options['path'] = app_path

        if deployed.buildPack:
            self.options['buildpack'] = " -b {} ".format(deployed.buildPack)
        if deployed.command:
            self.options['command'] = " -c {} ".format(deployed.command)
        if deployed.dockerImage:
            self.options['dockerImage'] = " --docker-image {} ".format(deployed.dockerImage)
        if deployed.dockerUsername:
            self.options['dockerUsername'] = " --docker-username {} ".format(deployed.dockerUsername)
        if deployed.noStart:
            self.options['noStart'] = " --no-start "
        if deployed.numInstances:
            self.options['numInstances'] = " -i {} ".format(deployed.numInstances)
        if deployed.disk:
            self.options['DISK'] = " -k {} ".format(deployed.disk)
        if deployed.memory:
            self.options['memory'] = " -m {} ".format(deployed.memory)
        if deployed.stack:
            self.options['stack'] = " -s {} ".format(deployed.stack)
        if deployed.noWait:
            self.options['noWait'] = " --no-wait "
        if deployed.endpoint:
            self.options['endpoint'] = " --endpoint {} ".format(deployed.endpoint)
        if deployed.healthTimeout:
            self.options['healthTimeout'] = " -t {} ".format(deployed.healthTimeout)
        if deployed.healthCheckType:
            self.options['healthCheckType'] = " -u {} ".format(deployed.healthCheckType)
        if deployed.route:
            if deployed.route == "hostname":
                self.options['route'] = " --hostname {} ".format(deployed.hostName)
            self.options['route'] = " --{} ".format(deployed.route)
        if deployed.domain:
            self.options['domain'] = " -d {} ".format(deployed.domain)
        if deployed.routePath:
            self.options['routePath'] = " --route-path {} ".format(deployed.routePath)
        if deployed.strategy:
            self.options['strategy'] = " --strategy {} ".format(deployed.strategy)
        return _convert_dict_to_cmd_opts(self.options) + " " + self.generate_variable(deployed)

    def route_options(self, deployed):
        routeOpts = ""
        if deployed.routeType == "http":
            if deployed.routeHostname:
                routeOpts += "--hostname {} ".format(deployed.routeHostname)
            if deployed.routePath:
                routeOpts += "--path {} ".format(deployed.routePath)
        elif deployed.routeType == "tcp":
            if deployed.port:
                routeOpts += "--port {} ".format(deployed.port)

        return routeOpts

    def service_options(self, deployed):
        serviceOpts = ""
        if deployed.plan:
            serviceOpts += "-p {} ".format(deployed.plan)
        if deployed.broker:
            serviceOpts += "-b {} ".format(deployed.broker)
        if deployed.configuration:
            serviceOpts += "-c {} ".format(deployed.configuration)
        if deployed.serviceTags.size != 0:
            serviceOpts += "-t {} ".format(deployed.serviceTags.join(', '))
        return serviceOpts

    def generate_login_command(self, apiEndpoint, username, password, organizationName, defaultSpaceName):
        return "{}/cf login -a {} -u {} -p {} -o {} -s {} --skip-ssl-validation".format(self.container.cfHome, apiEndpoint, username, password, organizationName, defaultSpaceName)

    def generate_logout_command(self):
        return self.container.cfHome + "/cf logout"

    def generate_delete_app_command(self, appName):
        return "{}/cf delete {} -f -r ".format(self.container.cfHome, appName)

    def create_route_command(self, routeDomain, routeOpts):
        if self.container.container.cliVersion == "v6":
            return "{}/cf create-route {} {} {} ".format(self.container.container.cfHome, self.container.spaceName, routeDomain, routeOpts)
        else:
            return "{}/cf create-route {} {} ".format(self.container.container.cfHome, routeDomain, routeOpts)

    def delete_route_command(self, routeDomain, routeOpts):
        return "{}/cf delete-route {} {} -f ".format(self.container.cfHome, routeDomain, routeOpts)

    def create_service_command(self, service, plan, serviceInstance, serviceOpts):
        return "{}/cf create-service {} {} {} {}".format(self.container.cfHome, service, plan, serviceInstance, serviceOpts)

    def update_service_command(self, serviceInstance, serviceOpts):
        return "{}/cf update-service {} {}".format(self.container.cfHome, serviceInstance, serviceOpts)

    def delete_service_command(self, serviceInstance):
        return "{}/cf delete-service {}".format(self.container.cfHome, serviceInstance)

def _convert_dict_to_cmd_opts(opts):
    if not opts:
        return ""
    cmd_opts = ""
    for k, v in opts.items():
        cmd_opts += v
    return cmd_opts


