from cf.app.helpers.app_helper import AppHelper
from cf.app.route.helpers.route_helper import RouteHelper


class AppStepsHelper(object):
    def __init__(self, context, steps, delta, space_name):
        self.__context = context
        self.__steps = steps
        self.__delta = delta
        self.__space_name = space_name

    def add_step_for_create_app(self, deployed):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Create app {} on {}".format(AppHelper.get_app_name(deployed),
                                                     self.__space_name),
            script="cf/app/create_app.py",
            jython_context={"deployed": deployed, "spaceName": self.__space_name},
            order=81
        ), self.__delta)

    def add_step_for_modify_app(self, deployed):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Update app {} on {}".format(AppHelper.get_app_name(deployed),
                                                     self.__space_name),
            script="cf/app/modify_app.py",
            jython_context={"deployed": deployed, "spaceName": self.__space_name},
            order=81
        ), self.__delta)

    def add_step_for_destroy_app(self, container, app_name, rolling_update=False):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Destroy app {} from {}".format(app_name, self.__space_name),
            script="cf/app/destroy_app.py",
            jython_context={"container": container, "spaceName": self.__space_name, "appName": app_name},
            order=99 if rolling_update else 19
        ), self.__delta)

    def add_step_for_bind_service(self, container, bound_service, app_name):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Bind app {} to service {}".format(app_name,
                                                           AppHelper.get_service_instance_name(bound_service)),
            script="cf/app/bind_app_to_service.py",
            jython_context={"container": container, "spaceName": self.__space_name, "boundService": bound_service,
                            "appName": app_name},
            order=85
        ), self.__delta)

    def add_step_for_unbind_service(self, container, bound_service, app_name, rolling_update=False):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Unbind app {} from service {}".format(app_name,
                                                               AppHelper.get_service_instance_name(
                                                                   bound_service)),
            script="cf/app/unbind_app_from_service.py",
            jython_context={"container": container, "spaceName": self.__space_name, "boundService": bound_service,
                            "appName": app_name},
            order=95 if rolling_update else 15
        ), self.__delta)

    def add_step_for_create_route(self, container, route):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Create route {} on {}".format(RouteHelper.get_route_url(route), self.__space_name),
            script="cf/app/route/create_route.py",
            jython_context={"container": container, "spaceName": self.__space_name, "route": route},
            order=83
        ), self.__delta)

    def add_step_for_destroy_route(self, container, route):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Destroy route {} on {}".format(RouteHelper.get_route_url(route),
                                                        self.__space_name),
            script="cf/app/route/destroy_route.py",
            jython_context={"container": container, "spaceName": self.__space_name, "route": route},
            order=17
        ), self.__delta)

    def add_step_for_bind_route(self, container, route, app_name):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Bind app {} to route {}".format(app_name,
                                                         RouteHelper.get_route_url(route)),
            script="cf/app/route/bind_app_to_route.py",
            jython_context={"container": container, "spaceName": self.__space_name, "route": route,
                            "appName": app_name},
            order=85
        ), self.__delta)

    def add_step_for_unbind_route(self, container, route, app_name, rolling_update=False):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Unbind app {} to route {}".format(app_name,
                                                           RouteHelper.get_route_url(route)),
            script="cf/app/route/unbind_app_from_route.py",
            jython_context={"container": container, "spaceName": self.__space_name, "route": route,
                            "appName": app_name},
            order=95 if rolling_update else 15
        ), self.__delta)

    def add_step_for_start_app(self, container, app_name):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Start app {} on {}".format(app_name, self.__space_name),
            script="cf/app/start_app.py",
            jython_context={"container": container, "spaceName": self.__space_name, "appName": app_name},
            order=86
        ), self.__delta)

    def add_step_for_rename_app(self, container, app_name, new_app_name):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Rename app {} to {}".format(app_name, new_app_name),
            script="cf/app/rename_app.py",
            jython_context={"container": container, "spaceName": self.__space_name, "appName": app_name,
                            "newAppName": new_app_name},
            order=80
        ), self.__delta)

    def add_step_for_stop_app(self, container, app_name, rolling_update=False):
        self.__context.addStepWithCheckpoint(self.__steps.jython(
            description="Stop app {} on {}".format(app_name, self.__space_name),
            script="cf/app/stop_app.py",
            jython_context={"container": container, "spaceName": self.__space_name, "appName": app_name},
            order=94 if rolling_update else 14
        ), self.__delta)

    def add_steps_for_unbinding_routes(self, previous_routes, routes, container, app_name, rolling_update=False):
        for route in previous_routes:
            if rolling_update:
                if not RouteHelper.contains_route(routes, route):
                    self.add_step_for_unbind_route(container, route, app_name)
                    self.add_step_for_destroy_route(container, route)
            else:
                self.add_step_for_unbind_route(container, route, app_name)
                if not RouteHelper.contains_route(routes, route):
                    self.add_step_for_destroy_route(container, route)

    def add_steps_for_binding_routes(self, previous_routes, routes, container, app_name):
        for route in routes:
            if not RouteHelper.contains_route(previous_routes, route):
                self.add_step_for_create_route(container, route)
            self.add_step_for_bind_route(container, route, app_name)
