#
# Copyright (c) 2020. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of Digital.ai, Inc. and its affiliates, subsidiaries, and licensors.
#

import json
from overtherepy import OverthereHostSession
from overtherepy import CommandResponse
from overtherepy import CapturingOverthereExecutionOutputHandler
import time
import sys
from overtherepy import CmdLine
import re


class HelmRunner:

    def __init__(self, helmclient, cluster):
        self.helmclient = helmclient
        self._preview = False

    def get_helm_command(self, install_or_upgrade=False):

        commandLine = CmdLine()
        commandLine.addArgument(self.helmclient.home+"/helm")
        #
        if self.helmclient.kubeContext is not None:
            commandLine.addArgument('--kube-context')
            commandLine.addArgument(self.helmclient.kubeContext)
        if self.helmclient.kubeConfig is not None:
            commandLine.addArgument('--kubeconfig')
            commandLine.addArgument(self.helmclient.kubeConfig)
        if self.helmclient.helmHost is not None:
            commandLine.addArgument('--host')
            commandLine.addArgument(self.helmclient.helmHost)
        if self.helmclient.debug:
            commandLine.addArgument('--debug')

        if install_or_upgrade:
            if self.helmclient.caFile is not None:
                commandLine.addArgument('--ca-file')
                commandLine.addArgument(self.helmclient.caFile)
            if self.helmclient.username is not None:
                commandLine.addArgument('--username')
                commandLine.addArgument(self.helmclient.username)
            if self.helmclient.password is not None:
                commandLine.addArgument('--password')
                commandLine.addPassword(self.helmclient.password);
        return commandLine


    def command_line(self, session, deployed):
        raise Exception("Not Implemented")

    # Masking credentials and secretInputVariables
    def mask_command_line(self, data, deployed=None):
        helm_password = "--password {0}".format(self.helmclient.password)
        datalist = data.getArguments();
        if helm_password in datalist:
            helm_password_arr = helm_password.split(" ")
            helm_password_arr[1] = "*******"
            new_helm_password = " ".join(helm_password_arr)
            data = data.replace(helm_password, new_helm_password)
        if deployed:
            secret_variables = ["{0}={1}".format(k,v) for k, v in deployed.secretInputVariables.items()]
            for secret_variable in secret_variables:
                if secret_variable in datalist:
                    secret_variable_arr = secret_variable.split("=")
                    secret_variable_arr[1] = "*******"
                    new_secret_variable = " ".join(secret_variable_arr)
                    data = data.replace(secret_variable, new_secret_variable)

        return data

    # Masking response
    def mask_response(self, data, deployed=None):
        if deployed:
            lines = data.split("\n")
            for i, line in enumerate(lines):
                if line.lower().__contains__('password:'):
                   if(line.strip().__contains__(": ")):
                        password_varriable_arr = line.strip().split(": ")
                        password_varriable_arr[1] = "*******"
                        new_password_varriable_arr = ": ".join(password_varriable_arr)
                        data = data.replace(line, line.replace(line.strip(), new_password_varriable_arr))

        return data

    def preview(self, deployed):
        session = OverthereHostSession(self.helmclient.host, stream_command_output=False)
        try:
            self._preview = True
            command_line = self.command_line(session, deployed)
            masked_command_line = self.mask_command_line(command_line, deployed)
            print(masked_command_line)
        finally:
            session.close_conn()

    def execute(self, deployed):
        session = OverthereHostSession(self.helmclient.host, stream_command_output=False)
        try:
            # command_line = CmdLine()
            command_line = self.command_line(session, deployed)
            print(command_line)
            uploaded_runner = session.upload_text_content_to_work_dir(
                command_line, 'xldeploy_helm.sh', executable=True)
            print(uploaded_runner.path)
            capture_so_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
            capture_se_handler = CapturingOverthereExecutionOutputHandler.capturingHandler()
            resp = session.get_conn().execute(capture_so_handler, capture_se_handler, command_line)
            # wait for output to drain
            time.sleep(1)
            response = CommandResponse(rc=resp, stdout=capture_so_handler.outputLines,
                                       stderr=capture_se_handler.outputLines)
            print(self.mask_response(str("\n".join(response.stdout)), deployed))
            print(self.mask_response(str("\n".join(response.stderr)), deployed))
            rc = response.rc
            if response.rc > 0:
                sys.exit(rc)
        finally:
            session.close_conn()

    def generate_variable(self, deployed):
        commandLineVariable = CmdLine()
        for k, v in deployed.inputVariables.items():
            commandLineVariable.addArgument("--set")
            commandLineVariable.addArgument(k+'='+v)
        for ki, vi in deployed.secretInputVariables.items():
            commandLineVariable.addArgument('--set')
            commandLineVariable.addPassword(ki+'='+vi)
        return commandLineVariable


    def namespace(self, deployed):
        return deployed.container.namespaceName if deployed.container.hasProperty("namespaceName") else deployed.container.projectName


class HelmInstall(HelmRunner):
    def command_line(self, session, deployed):
        commandLineInstall = CmdLine()
        commandLineInstall.add(self.get_helm_command(install_or_upgrade=True).getArguments())
        commandLineInstall.addArgument('upgrade')
        commandLineInstall.addArgument('--install')
        commandLineInstall.add(self.parameters(session, deployed).getArguments())
        if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3):
            commandLineInstall.addArgument("--insecure-skip-tls-verify")
        return commandLineInstall


    def parameters(self, session, deployed):
        values = {'chartName': deployed.chartName,
                  'namespace': self.namespace(deployed),
                  'name': deployed.name,
                  'chartVersion': deployed.chartVersion}

        commandLineInstallVariable = CmdLine()
        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            commandLineInstallVariable.addArgument(values.get('name'))
            commandLineInstallVariable.addArgument(values.get('chartName'))
            commandLineInstallVariable.addArgument('--namespace')
            commandLineInstallVariable.addArgument(values.get('namespace'))
            commandLineInstallVariable.addArgument('--version')
            commandLineInstallVariable.addArgument(values.get('chartVersion'))
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        commandLineInstallVariable.add(self.generate_variable(deployed).getArguments())

        for cf in deployed.configurationFiles:
            uploaded_file = session.upload_file_to_work_dir(cf.getFile())
            commandLineInstallVariable.addArgument("-f")
            commandLineInstallVariable.addArgument(uploaded_file.getPath())

        return commandLineInstallVariable


class HelmUpgrade(HelmRunner):

    def command_line(self, session, deployed):
        commandLineUpgrade = CmdLine()
        commandLineUpgrade.add(self.get_helm_command(install_or_upgrade=True).getArguments())
        commandLineUpgrade.addArgument('upgrade')
        commandLineUpgrade.add(self.parameters(session, deployed).getArguments())
        if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3):
            commandLineUpgrade.addArgument("--insecure-skip-tls-verify")
        return commandLineUpgrade

    def parameters(self, session, deployed):
        values = {'chartName': deployed.chartName,
                  'namespace': self.namespace(deployed),
                  'name': deployed.name,
                  'chartVersion': deployed.chartVersion}

        commandLineUpgradeVariable = CmdLine()
        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            commandLineUpgradeVariable.addArgument(values.get('name'))
            commandLineUpgradeVariable.addArgument(values.get('chartName'))
            commandLineUpgradeVariable.addArgument('--namespace')
            commandLineUpgradeVariable.addArgument(values.get('namespace'))
            commandLineUpgradeVariable.addArgument('--version')
            commandLineUpgradeVariable.addArgument(values.get('chartVersion'))
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        commandLineUpgradeVariable.add(self.generate_variable(deployed).getArguments())

        for cf in deployed.configurationFiles:
            uploaded_file = session.upload_file_to_work_dir(cf.getFile())
            commandLineUpgradeVariable.addArgument("-f")
            commandLineUpgradeVariable.addArgument(uploaded_file.getPath())

        return commandLineUpgradeVariable


class HelmAddRepo(HelmRunner):

    def command_line(self, session, deployed):
        commandLineAddRepo = CmdLine()
        commandLineAddRepo.add(self.get_helm_command(install_or_upgrade=True).getArguments())
        commandLineAddRepo.add(self.parameters(session, deployed).getArguments())
        if self.helmclient.forceUpdate:
            commandLineAddRepo.addArgument('--force-update')
        if (self.helmclient.insecureConnection and int(self.helmclient.version) == 3):
            commandLineAddRepo.addArgument("--insecure-skip-tls-verify")
        return commandLineAddRepo

    def parameters(self, session, deployed):
        commandLineAddRepoparameters = CmdLine()
        values = {'chartName': deployed.chartName.split('/')[0],
                  'repositoryUrl': deployed.repositoryUrl}

        if int(self.helmclient.version) == 2 or int(self.helmclient.version) == 3:
            commandLineAddRepoparameters.addArgument("repo")
            commandLineAddRepoparameters.addArgument("add")
            commandLineAddRepoparameters.addArgument(values.get('chartName'))
            commandLineAddRepoparameters.addArgument(values.get('repositoryUrl'))
        else:
            raise Exception("Unknown helm version {0}".format(self.helmclient.version))

        return commandLineAddRepoparameters

class HelmTest(HelmRunner):

    def command_line(self, session, deployed):
        commandLineHelmTest = CmdLine()
        if deployed.container.hasProperty('namespaceName'):
            namespaceName = deployed.container.namespaceName
        else:
            namespaceName = deployed.container.projectName
        commandLineHelmTest.add(self.get_helm_command(install_or_upgrade=False).getArguments())
        commandLineHelmTest.addArgument("test")
        commandLineHelmTest.addArgument("--namespace")
        commandLineHelmTest.addArgument(namespaceName)
        commandLineHelmTest.addArgument(deployed.name)
        return commandLineHelmTest
