from kubernetes import client
from kubernetes.client.api_client import ApiClient
import cPickle as pk
from commons.common_utils import CommonUtils


class PodHelper(object):
    def read_pod(self, deployed_pod, pod=client.V1Pod()):
        self.__apply_properties(pod, deployed_pod)
        pod_dict = ApiClient().sanitize_for_serialization(pod)
        volumes = PodHelper.__get_volume_dict(deployed_pod)
        pod_dict['spec']['volumes'] = volumes
        return pod_dict

    @staticmethod
    def get_pod_name(deployed):
        return deployed.podName if deployed.podName else deployed.name

    @staticmethod
    def __read_container(deployed_container):
        container = client.V1Container()
        container.image = deployed_container.image
        container.command = deployed_container.command
        container.args = deployed_container.args
        container.name = deployed_container.containerName
        if deployed_container.probes:
            for container_probe in deployed_container.probes:
                if container_probe.probeType == 'Readiness':
                    container.readiness_probe = PodHelper.__read_container_probe(container_probe)
                else:
                    container.liveness_probe = PodHelper.__read_container_probe(container_probe)
        return container

    @staticmethod
    def __read_container_probe(deployed_probe):
        probe = client.V1Probe()
        PodHelper.__apply_common_probe_properties(deployed_probe, probe)
        if deployed_probe.probeActionType == 'Exec':
            probe._exec = client.V1ExecAction(command=deployed_probe.command)
        elif deployed_probe.probeActionType == 'HTTPGet':
            probe.http_get = PodHelper.__read_http_action_probe(deployed_probe)
        elif deployed_probe.probeActionType == 'TCPSocket':
            probe.tcp_socket = client.V1TCPSocketAction(port=CommonUtils.convert_to_int_if_possible(deployed_probe.tcpPort))
        else:
            raise Exception("Probe type {} not supported".format(deployed_probe.probeActionType))
        return probe

    @staticmethod
    def __read_http_action_probe(deployed_probe):
        http_get_action = client.V1HTTPGetAction()
        http_get_action.host = deployed_probe.host
        http_get_action.port = CommonUtils.convert_to_int_if_possible(deployed_probe.port)
        http_get_action.path = deployed_probe.path
        http_get_action.scheme = deployed_probe.scheme
        if deployed_probe.httpHeaders:
            http_get_action.http_headers = []
            for k, v in deployed_probe.httpHeaders.iteritems():
                http_get_action.http_headers.append(client.V1HTTPHeader(name=k, value=v))
        return http_get_action

    @staticmethod
    def __apply_common_probe_properties(deployed_probe, probe):
        probe.initial_delay_seconds = deployed_probe.initialDelaySeconds
        probe.timeout_seconds = deployed_probe.timeoutSeconds
        probe.failure_threshold = deployed_probe.failureThreshold
        probe.success_threshold = deployed_probe.successThreshold
        probe.period_seconds = deployed_probe.periodSeconds

    @staticmethod
    def __read_container_port(deployed_port):
        port = client.V1ContainerPort()
        port.host_port = deployed_port.hostPort
        port.container_port = deployed_port.containerPort
        port.protocol = deployed_port.protocol
        port.host_ip = deployed_port.hostIP
        return port

    @staticmethod
    def __read_volume_mount(deployed_volume_mount):
        volume_mount = client.V1VolumeMount()
        volume_mount.mount_path = deployed_volume_mount.mountPath
        volume_mount.name = deployed_volume_mount.volumeName if bool(
            deployed_volume_mount.volumeName) else deployed_volume_mount.name
        volume_mount.read_only = deployed_volume_mount.readOnly
        volume_mount.sub_path = deployed_volume_mount.subPath
        return volume_mount

    def __apply_properties(self, pod, deployed_pod):
        pod.metadata = client.V1ObjectMeta(name=self.get_pod_name(deployed_pod))
        if deployed_pod.labels:
            pod.metadata.labels = deployed_pod.labels

        spec = client.V1PodSpec()
        spec.host_network = deployed_pod.hostNetwork
        spec.containers = []
        for deployed_container in deployed_pod.containers:
            container = PodHelper.__read_container(deployed_container)
            if deployed_container.ports:
                container.ports = []
                for deployed_port in deployed_container.ports:
                    port = PodHelper.__read_container_port(deployed_port)
                    container.ports.append(port)

            if deployed_container.volumeBindings:
                container.volume_mounts = []
                for volume_binding in deployed_container.volumeBindings:
                    volume_mount = self.__read_volume_mount(volume_binding)
                    container.volume_mounts.append(volume_mount)
            spec.containers.append(container)
        pod.spec = spec

    @staticmethod
    def __get_volume_dict(deployed_pod):
        volumes = []
        for deployed_volume in deployed_pod.volumes:
            volume = {'name': deployed_volume.volumeName if bool(deployed_volume.volumeName) else deployed_volume.name}
            vol_prop = {}
            for key, val in deployed_volume.properties.items():
                vol_prop[key] = True if val.lower() == 'true' else pk.loads(pk.dumps(val))

            volume[deployed_volume.volumeType] = vol_prop
            volumes.append(volume)
        return volumes

    @staticmethod
    def validate_pod(deployed_pod):
        if deployed_pod.containers:
            for deployed_container in deployed_pod.containers:
                PodHelper.__validate_container(deployed_container)

    @staticmethod
    def __validate_container(deployed_container):
        if deployed_container.probes:
            PodHelper.__validate_probes(deployed_container.probes, 'Readiness', deployed_container.containerName)
            PodHelper.__validate_probes(deployed_container.probes, 'Liveness', deployed_container.containerName)

    @staticmethod
    def __validate_probes(all_probes, probe_type, container_name):
        probes = [probe for probe in all_probes if probe.probeType == probe_type]
        if len(probes) > 1:
            raise Exception("Maximum of 2 probes, one each for readiness and liveness, can be added for each container, but found {} of type {} for container {}"
                            .format(len(probes), probe_type, container_name))
