#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#

from kubernetes import client
from kubernetes.client.api_client import ApiClient
from xld.kubernetes.pod.pod_helper import PodHelper


class CronJobHelper(object):
    def __init__(self):
        self.__pod_helper = PodHelper()

    def read_cronjob(self, deployed_cronjob):
        cronjob = client.V1beta1CronJob()
        cronjob.metadata = client.V1ObjectMeta(name=self.get_cronjob_name(deployed_cronjob))

        pod_template = self.__pod_helper.read_pod(deployed_pod=deployed_cronjob, pod=client.V1PodTemplateSpec())
        pod_template['metadata']['name'] = cronjob.metadata.name

        job_template = self.read_job_template(deployed_cronjob, pod_template)

        spec = client.V1beta1CronJobSpec(concurrency_policy=deployed_cronjob.concurrencyPolicy,
                                          job_template=job_template,
                                          schedule=deployed_cronjob.schedule, suspend=deployed_cronjob.suspend)

        if deployed_cronjob.failedJobsHistoryLimit:
            spec.failed_jobs_history_limit = deployed_cronjob.failedJobsHistoryLimit
        if deployed_cronjob.successfulJobsHistoryLimit:
            spec.successful_jobs_history_limit = deployed_cronjob.successfulJobsHistoryLimit

        if deployed_cronjob.startingJobsDeadlineSeconds:
            spec.starting_deadline_seconds = deployed_cronjob.startingJobsDeadlineSeconds

        spec = ApiClient().sanitize_for_serialization(spec)
        cronjob = ApiClient().sanitize_for_serialization(cronjob)
        cronjob['spec'] = spec
        return cronjob

    def read_job_template(self, deployed_cronjob, pod_template):
        job_template = client.V1beta1JobTemplateSpec()
        job_template.metadata = client.V1ObjectMeta(name=self.get_cronjob_name(deployed_cronjob))
        if deployed_cronjob.labels:
            job_template.metadata.labels = deployed_cronjob.labels

        spec = client.V1JobSpec(template=pod_template,
                                backoff_limit=deployed_cronjob.backoffLimit)
        if deployed_cronjob.completions:
            spec.completions = deployed_cronjob.completions
        if deployed_cronjob.parallelism:
            spec.parallelism = deployed_cronjob.parallelism
        if deployed_cronjob.activeJobsDeadlineSeconds:
            spec.active_deadline_seconds = deployed_cronjob.activeJobsDeadlineSeconds

        job_template.spec = spec
        job_template = ApiClient().sanitize_for_serialization(job_template)
        return job_template

    @staticmethod
    def get_cronjob_name(deployed):
        return deployed.cronJobName if deployed.cronJobName else deployed.name

    @staticmethod
    def validate_cronjob(deployed_cronjob):
        if deployed_cronjob.activeJobsDeadlineSeconds and deployed_cronjob.activeJobsDeadlineSeconds < 0:
            raise RuntimeError("Active Jobs Deadline Seconds should be a positive number.")
