#
# Copyright (c) 2018. All rights reserved.
#
# This software and all trademarks, trade names, and logos included herein are the property of XebiaLabs, Inc. and its affiliates, subsidiaries, and licensors.
#
import textwrap

from xld.kubernetes.factories.handler_factory import ResourceFactoryResolver, ContainerHelperFactory
from xld.kubernetes.resource.describe_helpers import DescribeHelper


class WaitHelper(object):
    def __init__(self, deployed, data, context):
        self.__deployed = deployed
        self.__data = data
        self.__container_helper = ContainerHelperFactory(deployed.container).create()
        self.__describe_helper = DescribeHelper(self.__deployed)
        self.__context = context
        self.__keyReadinessProbeRetryCounter = "{0}-{1}-{2}".format(self.__deployed, self.__data['metadata']['name'], self.__data['kind'])

    def __get_readinessProbeRetryCounter(self):
        if self.__context.getAttribute(self.__keyReadinessProbeRetryCounter ) is None:
            self.__context.setAttribute(self.__keyReadinessProbeRetryCounter ,self.__deployed.readinessProbeRetry)
        return self.__context.getAttribute(self.__keyReadinessProbeRetryCounter )

    def __set_readinessProbeRetryCounter(self, value):
        return self.__context.setAttribute(self.__keyReadinessProbeRetryCounter ,value )

    def wait_for_pods(self, result, action):
        with ResourceFactoryResolver(self.__deployed).get_factory().get(self.__data) as provider:
            resources = provider.filter_resources_by_definition(
                namespace=self.__container_helper.get_container_name(self.__deployed.container),
                resource_definition=self.__data)

        metaname = self.__data['metadata']['name']
        item_kind = self.__data['kind']
        action = action.lower()

        print("Waiting for {0} {1} to be ready\n".format(item_kind, metaname))

        for item in resources.items:
            print("#####")
            kind = resources.kind.replace("List", "")
            print("{0} Status:".format(kind))
            print("- Replica(s) wanted: %s" % item.spec.replicas)
            print("- readyReplicas: %s" % item.status.readyReplicas)
            print("- updatedReplicas: %s" % item.status.updatedReplicas)
            print("- unavailableReplicas: %s" % item.status.unavailableReplicas)
            print("- availableReplicas: %s" % item.status.availableReplicas)
            if item.status.unavailableReplicas > 0 or \
                    (action == "modify" and item.status.currentRevision != item.status.updateRevision) or \
                    item.status.readyReplicas != item.spec.replicas:
                if self.__get_readinessProbeRetryCounter()  > 0:
                    print("Waiting for {0} {1} to be in running state, retries left: {2} (Approx {3} seconds)".format(
                        item_kind, metaname, self.__get_readinessProbeRetryCounter(),
                        self.__get_readinessProbeRetryCounter() * 5))
                    self.__set_readinessProbeRetryCounter(self.__get_readinessProbeRetryCounter() - 1)
                    result = "RETRY"
                else:
                    self.__describe_helper.describe_pod(metaname, item_kind)
                    self.process_fail(item_kind)
            else:
                # Print logs
                self.__describe_helper.describe_pod(metaname, item_kind)
        return result

    def wait_for_resource(self, result, display_resource_on_logs):

        with ResourceFactoryResolver(self.__deployed).get_factory().get(self.__data) as provider:
            resources = provider.filter_resources_by_definition(
                namespace=self.__container_helper.get_container_name(self.__deployed.container),
                resource_definition=self.__data)

        metaname = self.__data['metadata']['name']
        kind = self.__data['kind']

        print("Waiting for {0} {1} to be ready".format(kind, metaname))

        if resources.kind == "PodList":
            retry_count = 0
            starting_pods = [pod for pod in resources.items if pod.status.phase != "Running"]

            if len(starting_pods) > 0 and retry_count < self.__deployed.resourceWaitTime:
                print("Waiting for Pod to be activated ...")
                result = "RETRY"
                retry_count += 1

        elif resources.kind == "RouteList":
            for item in resources.items:
                for ingress in item.status.ingress:
                    for condition in ingress.conditions:
                        if condition.type == "Admitted" and condition.status == "Unknown" and retry_count < self.__deployed.resourceWaitTime:
                            print("Waiting for Route to be activated ...")
                            result = "RETRY"
                            retry_count = retry_count + 1

        elif resources.kind == "DeploymentConfigList":
            for item in resources.items:
                if (self.__deployed.validatePodsCreation):
                    result = self.process_item(item, self.__deployed, resources)

        elif resources.kind in ["ConfigMapList", "SecretList", "CronJobList", "ServiceList"]:
            if not bool(resources.items) and self.__deployed.resourceWaitTime > 0:
                print("Waiting for {0} to be ready ...".format(resources.kind.replace("List", "")))
                result = "RETRY"
                self.__deployed.resourceWaitTime -= 1
            else:
                if resources.kind == "ServiceList" and self.__deployed.waitServiceLoadBalancerIngress:
                    for service in resources.items:
                        if service.spec.type == "LoadBalancer":
                            if not hasattr(service.status.loadBalancer,
                                           'ingress') or service.status.loadBalancer.ingress is None:
                                print("Waiting for LoadBalancer Ingress of Service '{0}' to be ready ...".format(
                                    service.metadata.name))
                                result = "RETRY"
                                self.__deployed.resourceWaitTime -= 1


        elif resources.kind == 'PersistentVolumeClaimList':
            for pv_claim in resources.items:
                if pv_claim.status.phase != 'Bound' and self.__deployed.resourceWaitTime > 0:
                    print("Waiting for PersistentVolumeClaim to be in Bound phase ...")
                    result = "RETRY"
                    self.__deployed.resourceWaitTime -= 1

        if display_resource_on_logs:
            print(textwrap.dedent("""
            ---------------------------------------
            {0} {1} Metadata
            ---------------------------------------
            """.format(kind, metaname)))
            print("{0}".format(resources))

        return result

    def process_item(self, item, __deployed, resources):
        for trigger in item.spec.triggers:
            if trigger.type == "ConfigChange" or (trigger.imageChangeParams and trigger.imageChangeParams.automatic):
                print("#####")
                kind = resources.kind.replace("List", "")
                print("{0} Status:".format(kind))
                print("- Replica(s) wanted: %s" % item.spec.replicas)
                print("- readyReplicas: %s" % item.status.readyReplicas)
                print("- updatedReplicas: %s" % item.status.updatedReplicas)
                print("- unavailableReplicas: %s" % item.status.unavailableReplicas)
                print("- availableReplicas: %s" % item.status.availableReplicas)
                if item.status.availableReplicas != item.spec.replicas or item.status.unavailableReplicas > 0:
                    if int(__deployed.readinessProbeRetry) > 0:
                        print(
                            "Waiting for {3}'s DeploymentConfig's {0} to have pods in running state, retries left: {1} (Approx {2} seconds)".format(
                                item.metadata.name,
                                __deployed.readinessProbeRetry,
                                __deployed.readinessProbeRetry * 5,
                                kind))
                        __deployed.readinessProbeRetry -= 1
                        return "RETRY"

                    else:
                        self.process_fail("pods")
                return "SUCCESS"

    # hack method to fail process if pods are not at a ready status. result 'FAIL' does not work
    def process_fail(self, type):
        print("None of the {0} were at a ready status after reaching the max number of retries. \n".format(type))
        # Get the logs form the pod
        raise Exception(
            "Max number of retries reached. Please verify if readiness probe check is correct or increase the number 'Readiness Probe Retry' in ci properties... \n")
        # fails the deployment here
        exit(1)
